#include <iostream>
#include "Logger.h"
#include "gradients.h"
#include "hoverpoints.h"

using namespace std;

ShadeWidget::ShadeWidget(ShadeType type, QWidget *parent)
	: QWidget(parent), m_shade_type(type), m_alpha_gradient(QLinearGradient(0, 0, 0, 0))
{

	// draw black/white checkers background for alpha shade
	if (m_shade_type == ARGBShade) {
		QPixmap pm(20, 20);
		QPainter pmp(&pm);
		pmp.fillRect(0, 0, 10, 10, Qt::lightGray);
		pmp.fillRect(10, 10, 10, 10, Qt::lightGray);
		pmp.fillRect(0, 10, 10, 10, Qt::darkGray);
		pmp.fillRect(10, 0, 10, 10, Qt::darkGray);
		pmp.end();
		QPalette pal = palette();
		pal.setBrush(backgroundRole(), QBrush(pm));
		setAutoFillBackground(true);
		setPalette(pal);

	} else {
		setAttribute(Qt::WA_NoBackground);

	}

	QPolygonF points;
	points << QPointF(0, sizeHint().height())
		<< QPointF(sizeHint().width(), 20);

	m_hoverPoints = new HoverPoints(this, HoverPoints::CircleShape);
	m_hoverPoints->setConnectionType(HoverPoints::LineConnection);
	m_hoverPoints->setPoints(points);
	m_hoverPoints->setPointLock(0, HoverPoints::LockToLeft);
	m_hoverPoints->setPointLock(1, HoverPoints::LockToRight);

	m_hoverPoints->setSortType(HoverPoints::XSort);
	m_hoverPoints->firePointChange();

	setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);

	connect(m_hoverPoints, SIGNAL(pointsChanged(QPolygonF)), this, SIGNAL(colorsChanged()));
}

QPolygonF ShadeWidget::points() const
{
	return m_hoverPoints->points();
}

void ShadeWidget::setPoints(QPolygonF points)
{
    m_hoverPoints->setPoints(points);
}

uint ShadeWidget::colorAt(int x)
{
	generateShade();

	QPolygonF pts = m_hoverPoints->points();

	for (int i=1; i < pts.size(); ++i) {
		if (pts.at(i-1).x() <= x && pts.at(i).x() >= x) {
			QLineF l(pts.at(i-1), pts.at(i));
			QPointF p1 = pts.at(i-1);
			QPointF p2 = pts.at(i);
			float factor;

			if (p2.x() != p1.x())
			    factor = (x - p1.x()) / (p2.x() - p1.x());
			else
			    factor = 0.0;

			//cout << p1.x() << ":" << p2.x() << ":" << factor << endl;
			QPointF p = l.pointAt(factor);

			l.setLength(l.length() * ((x - l.x1()) / l.dx()));
			//return m_shade.pixel(x, p1.y() + ((p2.y() - p1.y()) / (p2.x() - p1.x())) * (x - p1.x()));
			/*return m_shade.pixel(qRound(qMin(l.x2(), (qreal(m_shade.width() - 1)))),
				qRound(qMin(l.y2(), qreal(m_shade.height() - 1))));*/
			return m_shade.pixel(qMin(qRound(p.x()), m_shade.width() - 1), qMin(qRound(p.y()), m_shade.height() - 1));
		}
	}
	return 0;
}


void ShadeWidget::setGradientStops(const QGradientStops &stops)
{
	if (m_shade_type == ARGBShade) {
		m_alpha_gradient = QLinearGradient(0, 0, width(), 0);

		for (int i=0; i<stops.size(); ++i) {
			QColor c = stops.at(i).second;
			m_alpha_gradient.setColorAt(stops.at(i).first, QColor(c.red(), c.green(), c.blue()));
		}

		m_shade = QImage();
		generateShade();
		update();
	}
}


void ShadeWidget::paintEvent(QPaintEvent *)
{
	generateShade();


	if (m_shade_type == ARGBShade)
	{
	    QPainter p(this);
	    p.drawImage(0,0, m_histogram);

	    p.setPen(QColor(146, 146, 146));
	    p.drawRect(0, 0, width() - 1, height() - 1);
	}
	else
	{
	    QPainter p(this);
	    p.drawImage(0, 0, m_shade);

	    p.setPen(QColor(146, 146, 146));
	    p.drawRect(0, 0, width() - 1, height() - 1);
	}

}


void ShadeWidget::generateShade()
{
	if (m_shade.isNull() || m_shade.size() != size()) {

		if (m_shade_type == ARGBShade) {
		    {
			m_shade = QImage(size(), QImage::Format_ARGB32_Premultiplied);
			m_shade.fill(0);

			QPainter p(&m_shade);
			p.fillRect(rect(), m_alpha_gradient);

			p.setCompositionMode(QPainter::CompositionMode_DestinationIn);
			QLinearGradient fade(0, 0, 0, height());
			fade.setColorAt(0, QColor(0, 0, 0, 255));

			fade.setColorAt(0.95, QColor(0, 0, 0, 0));
			p.fillRect(rect(), fade);
		    }
		    {
			//now draw histogram

			m_histogram = QImage(size(), QImage::Format_ARGB32_Premultiplied);
			m_histogram.fill(0);

			QPainter p(&m_histogram);
			p.fillRect(rect(), m_alpha_gradient);

			p.setCompositionMode(QPainter::CompositionMode_DestinationIn);
			QLinearGradient fade(0, 0, 0, height());
			fade.setColorAt(0, QColor(0, 0, 0, 255));

			fade.setColorAt(0.95, QColor(0, 0, 0, 0));
			p.fillRect(rect(), fade);

/*			float *histodata = VolumeData::instance()->getLogNormalizedHistogramData();
			if (histodata == NULL)
			    return;

			QPen pen;
			p.setCompositionMode(QPainter::CompositionMode_Source);

			for (int i=0; i < width(); i++)
			{
			    pen.setColor(QColor(0,0,0));
			    pen.setWidth(1);

			    p.setPen(pen);
			    p.drawPoint(i, height() - 1 - height() * histodata[(int)(4096.0f * ((float)i / width()))]);

			    pen.setColor(QColor(50,50,50));
			    p.setPen(pen);
			    p.drawPoint(i, height() - height() * histodata[(int)(4096.0f * ((float)i / width()))]);
			    p.drawPoint(i, height() - 2 - height() * histodata[(int)(4096.0f * ((float)i / width()))]);

			    pen.setColor(QColor(130,130,130));
			    p.setPen(pen);
			    p.drawPoint(i, height() + 1 - height() * histodata[(int)(4096.0f * ((float)i / width()))]);
			    p.drawPoint(i, height() - 3 - height() * histodata[(int)(4096.0f * ((float)i / width()))]);
			}*/
		    }

		} else {
			m_shade = QImage(size(), QImage::Format_RGB32);
			QLinearGradient shade(0, 0, 0, height());
			shade.setColorAt(0.95, Qt::black);

			if (m_shade_type == RedShade)
				shade.setColorAt(0, Qt::red);
			else if (m_shade_type == GreenShade)
				shade.setColorAt(0, Qt::green);
			else
				shade.setColorAt(0, Qt::blue);

			QPainter p(&m_shade);
			p.fillRect(rect(), shade);
		}
	}


}


GradientEditor::GradientEditor(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *vbox = new QVBoxLayout(this);
	vbox->setSpacing(1);
	vbox->setMargin(1);

	m_red_shade = new ShadeWidget(ShadeWidget::RedShade, this);
	m_green_shade = new ShadeWidget(ShadeWidget::GreenShade, this);
	m_blue_shade = new ShadeWidget(ShadeWidget::BlueShade, this);
	//m_alpha_shade = new ShadeWidget(ShadeWidget::ARGBShade, this);

	vbox->addWidget(m_red_shade);
	vbox->addWidget(m_green_shade);
	vbox->addWidget(m_blue_shade);
	//vbox->addWidget(m_alpha_shade);

	connect(m_red_shade, SIGNAL(colorsChanged()), this, SLOT(pointsUpdated()));
	connect(m_green_shade, SIGNAL(colorsChanged()), this, SLOT(pointsUpdated()));
	connect(m_blue_shade, SIGNAL(colorsChanged()), this, SLOT(pointsUpdated()));
	//connect(m_alpha_shade, SIGNAL(colorsChanged()), this, SLOT(pointsUpdated()));
}

QColor GradientEditor::getColorAt(int i)
{
	//TODO: maybe we should interpolate? cm, 7.11.2010

	double x_between_0_and_1 = i / 4096.0;

	int x = m_red_shade->width() * x_between_0_and_1;

	int shade_width = m_red_shade->width();
	float var;

	if (i != 0)
		var = 1.0f / (i % (4096 / shade_width));
	else
		var = 0.0f;

	if (var > 1.0f)
		var = 0.0f;

	int x2 = m_red_shade->width() * x_between_0_and_1 + 1;

	if (x2 >= shade_width)
		x2 = shade_width-1;

	//write QColor object. has to be translated again before it can be used in the transfer function
	QColor color1((0x00ff0000 & m_red_shade->colorAt(int(x))) >> 16,
		(0x0000ff00 & m_green_shade->colorAt(int(x))) >> 8,
		(0x000000ff & m_blue_shade->colorAt(int(x)))/*,
		(0xff000000 & m_alpha_shade->colorAt(int(x))) >> 24*/);

	return color1;
}

inline static bool x_less_than(const QPointF &p1, const QPointF &p2)
{
	return p1.x() < p2.x();
}

void GradientEditor::saveToFile(QString filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (!file.open(QIODevice::WriteOnly))
	cerr << "something went wrong" << endl;

    QPolygonF redPoints = this->m_red_shade->points();
    QPolygonF greenPoints = this->m_green_shade->points();
    QPolygonF bluePoints = this->m_blue_shade->points();
    //QPolygonF alphaPoints = this->m_alpha_shade->points();

    //first, write red points
    out << redPoints.size() << endl;

    for (int i=0; i < redPoints.size(); i++)
	    out << redPoints[i].x() / m_red_shade->width() << " " << redPoints[i].y() / m_red_shade->height() << endl;

    //then write green points

    out << greenPoints.size() << endl;

    for (int i=0; i < greenPoints.size(); i++)
	    out << greenPoints[i].x() / m_green_shade->width() << " " << greenPoints[i].y() / m_green_shade->height() << endl;

    //blue points
    out << bluePoints.size() << endl;

    for (int i=0; i < bluePoints.size(); i++)
	    out << bluePoints[i].x() / m_blue_shade->width() << " " << bluePoints[i].y() / m_blue_shade->height() << endl;

    //at last, alpha points

   // out << alphaPoints_vector.size() << endl;

    //for (int i=0; i < alphaPoints_vector.size(); i++)
	//    out << alphaPoints_vector[i].x() / m_alpha_shade->width() << " " << alphaPoints_vector[i].y() / m_alpha_shade->height() << endl;

    file.close();
}

void GradientEditor::loadFromFile(QString filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QPolygonF red_points, blue_points, green_points, alpha_points;

    if (!file.open(QIODevice::ReadOnly))
	cerr << "something went wrong" << endl;

    //first, read red points
    int red_points_count;
    in >> red_points_count;

    for (int i=0; i < red_points_count; i++)
    {
	float x, y;
	in >> x;
	in >> y;
	red_points.append(QPointF(x * m_red_shade->width(),y * m_red_shade->height()));
    }

    //then read green points
    int green_points_count;
    in >> green_points_count;

    for (int i=0; i < green_points_count; i++)
    {
	float x, y;
	in >> x;
	in >> y;
	green_points.append(QPointF(x * m_green_shade->width(),y * m_green_shade->height()));
    }

    //blue points
    int blue_points_count;
    in >> blue_points_count;

    for (int i=0; i < blue_points_count; i++)
    {
	float x, y;
	in >> x;
	in >> y;
	blue_points.append(QPointF(x * m_blue_shade->width(),y * m_blue_shade->height()));
    }

    //at last, alpha points
   /* int alpha_points_count;
    in >> alpha_points_count;

    for (int i=0; i < alpha_points_count; i++)
    {
	float x, y;
	in >> x;
	in >> y;
	alpha_points.append(QPointF(x * m_alpha_shade->width(),y * m_alpha_shade->height()));
    }*/

    file.close();

    m_red_shade->setPoints(red_points);
    m_green_shade->setPoints(green_points);
    m_blue_shade->setPoints(blue_points);
  //  m_alpha_shade->setPoints(alpha_points);
}

void GradientEditor::pointsUpdated()
{
	qreal w = m_red_shade->width();

	QGradientStops stops;

	QPolygonF points;

	points += m_red_shade->points();
	points += m_green_shade->points();
	points += m_blue_shade->points();
	//points += m_alpha_shade->points();

	qSort(points.begin(), points.end(), x_less_than);

	for (int i=0; i<points.size(); ++i) {
		qreal x = int(points.at(i).x());
		if (i < points.size() - 1 && x == points.at(i+1).x())
			continue;
		QColor color((0x00ff0000 & m_red_shade->colorAt(int(x))) >> 16,
			(0x0000ff00 & m_green_shade->colorAt(int(x))) >> 8,
			(0x000000ff & m_blue_shade->colorAt(int(x)))/*,
			(0xff000000 & m_alpha_shade->colorAt(int(x))) >> 24*/);

		if (x / w > 1)
			return;

		stops << QGradientStop(x / w, color);
	}

//	m_alpha_shade->setGradientStops(stops);


	emit gradientStopsChanged(stops);
	emit transferFunctionChanged(prepareTransferFunctionData());
}


unsigned char * GradientEditor::prepareTransferFunctionData()
{
	/* now prepare data for use as an opengl texture */

	//allocate memory (one byte for each color channel (r,g,b,a) for a 1d texture with width 4096)
	//attention: caller must free memory with delete[] to prevent a huge memory leak!!!
	unsigned char * p = new unsigned char[4 * 4096];

	//fill data
	for (int i=0; i < 4096; i++)
	{
		QColor color_for_index = this->getColorAt(i);

		//now fill buffer as RGBA and scale values to [0, 255]
		p[4*i + 0] = (unsigned char) (color_for_index.redF() * 255);
		p[4*i + 1] = (unsigned char) (color_for_index.greenF() * 255);
		p[4*i + 2] = (unsigned char) (color_for_index.blueF() * 255);
		p[4*i + 3] = (unsigned char) (color_for_index.alphaF() * 255);
	}

	return p;
}

static void set_shade_points(const QPolygonF &points, ShadeWidget *shade)
{
	//set points
	shade->hoverPoints()->setPoints(points);
	shade->hoverPoints()->setPointLock(0, HoverPoints::LockToLeft);
	shade->hoverPoints()->setPointLock(points.size() - 1, HoverPoints::LockToRight);
	shade->update();
}

void GradientEditor::setGradientStops(const QGradientStops &stops)
{
	//set points in all shades, decode stops in 4 channels

	QPolygonF pts_red, pts_green, pts_blue, pts_alpha;

	qreal h_red = m_red_shade->height();
	qreal h_green = m_green_shade->height();
	qreal h_blue = m_blue_shade->height();
	//qreal h_alpha = m_alpha_shade->height();

	for (int i=0; i<stops.size(); ++i) {
		qreal pos = stops.at(i).first;
		QRgb color = stops.at(i).second.rgba();
		pts_red << QPointF(pos * m_red_shade->width(), h_red - qRed(color) * h_red / 255);
		pts_green << QPointF(pos * m_green_shade->width(), h_green - qGreen(color) * h_green / 255);
		pts_blue << QPointF(pos * m_blue_shade->width(), h_blue - qBlue(color) * h_blue / 255);
		//pts_alpha << QPointF(pos * m_alpha_shade->width(), h_alpha - qAlpha(color) * h_alpha / 255);
	}

	set_shade_points(pts_red, m_red_shade);
	set_shade_points(pts_green, m_green_shade);
	set_shade_points(pts_blue, m_blue_shade);
	//set_shade_points(pts_alpha, m_alpha_shade);
}
