#ifndef FLOWRENDERER_H
#define FLOWRENDERER_H

#include <QtOpenGL>

/// Central class for rendering outputs.
class FlowRenderer : public QObject
{
    Q_OBJECT

private:
    GLuint m_gliArrowTexture;
    GLuint m_gliStreamlinesTexture;
    GLuint m_gliStreamlinesGlyphsTexture;
    GLuint m_gliGlyphTexture;

    bool m_bShowArrowTexture;
    bool m_bShowStreamlinesTexture;
    bool m_bAddStreamlines;
    bool m_bGLThingsInitialized;

    int m_iColorCoding;
    int m_iStreamlinesMode; ///< 0=Euler, 1=RK2, 2=RK4
    int m_iStreamlineWidth;
    int m_iStreamlineTexturingMode; ///< 0=off, 1=f1, 2=f2
    int m_iStreamlineTexturingPeriodLength; ///< length of the period (has to be >1)
    int m_iRenderWidth;
    int m_iRenderHeight;

    float m_fStreamlines_Alpha;
    float m_fStreamlines_dsep;
    float m_fStreamlines_dtest;
    float m_fStreamlines_dstep;
    int m_iMaxStreamlineSegments;
    bool m_bUseTapering;
    bool m_bUseGlyphs;

    QGLShaderProgram m_ShaderProgram;

    void checkInitialized();

public slots:
    void updateArrowTexture();
    void updateStreamlinesTexture();

    void render();

    void setColorCoding(int i) { m_iColorCoding = i; }
    void setShowArrows(bool b) { m_bShowArrowTexture = b; emit ShowArrowsChanged(b);}
    void setShowStreamlines(bool b) { m_bShowStreamlinesTexture = b;  emit ShowStreamlinesChanged(b); }
    void setStreamlinesMode(int i) {m_iStreamlinesMode = i;}
    void setDsep(double f) { m_fStreamlines_dsep = f; emit DsepChanged(f);}
    void setDtest(double f) { m_fStreamlines_dtest = f; emit DtestChanged(f); }
    void setDstep(double f) { m_fStreamlines_dstep = f; emit DstepChanged(f); }
    void setUseTapering(bool b) { m_bUseTapering = b; emit UseTaperingChanged(b); }
    void setStreamlineWidth(int i) { m_iStreamlineWidth = i; emit StreamlineWidthChanged(i); }
    void setRenderSize(int w, int h) { m_iRenderWidth = w; m_iRenderHeight = h; }
    void setStreamlineTexturingMode(int i) { m_iStreamlineTexturingMode = i; emit StreamlineTexturingModeChanged(i);}
    void setStreamlineTexturingPeriodLength(int i) { m_iStreamlineTexturingPeriodLength = i;  emit StreamlineTexturingPeriodLengthChanged(i);}
    void setUseGlyphs(bool b) { m_bUseGlyphs = b; emit UseGlyphsChanged(b); }
    void setMaxStreamlineSegments(int i) { m_iMaxStreamlineSegments = i; emit MaxStreamlineSegmentsChanged(i); }
    void setStreamlinesAlpha(int i) { m_fStreamlines_Alpha = i / 100.0f; emit StreamlinesAlphaChanged(i); }
    void setStreamlinesComboMode(int i) { m_bAddStreamlines = (i==0); emit StreamlinesComboModeChanged(i); }

signals:
    void ShowArrowsChanged(bool b);
    void ShowStreamlinesChanged(bool b);
    void DsepChanged(double f);
    void DtestChanged(double f);
    void DstepChanged(double f);
    void UseTaperingChanged(bool b);
    void UseGlyphsChanged(bool b);
    void StreamlineWidthChanged(int i);
    void StreamlineTexturingModeChanged(int i);
    void StreamlineTexturingPeriodLengthChanged(int i);
    void MaxStreamlineSegmentsChanged(int i);
    void StreamlinesComboModeChanged(int i);
    void StreamlinesAlphaChanged(int i);

public:
    static FlowRenderer *instance();
    FlowRenderer();
};

#endif // FLOWRENDERER_H
