#include <GL/glew.h>
#include <QColor>
#include <QPainter>
#include <QGLShaderProgram>
#include <QGLShader>
#include <QHash>
#include <QMatrix>
#include <qmath.h>

#include "FlowRenderer.h"
#include "TransferFunction.h"
#include "FlowData.h"
#include <assert.h>

FlowRenderer::FlowRenderer() : m_bGLThingsInitialized(false)
{

}

FlowRenderer* FlowRenderer::instance()
{
    static FlowRenderer singelton_instance;
    return &singelton_instance;
}

void FlowRenderer::checkInitialized()
{
    if (m_bGLThingsInitialized == false)
    {
	glGenTextures(1, &m_gliArrowTexture);
	glGenTextures(1, &m_gliStreamlinesTexture);
	glGenTextures(1, &m_gliGlyphTexture);

	m_ShaderProgram.addShaderFromSourceFile(QGLShader::Vertex, "shaders/simple.vs");
	m_ShaderProgram.addShaderFromSourceFile(QGLShader::Fragment, "shaders/render_channel_with_tf.fs");
	m_ShaderProgram.link();

	m_bGLThingsInitialized = true;
	this->setDsep(0.03);
	this->setDstep(0.015);
	this->setDtest(0.5);
	this->setStreamlineWidth(2);
	this->setUseGlyphs(true);
	this->setStreamlineTexturingMode(0);
	this->setStreamlineTexturingPeriodLength(10);
	this->setMaxStreamlineSegments(25);
	this->setUseTapering(true);
	this->setShowArrows(false);
	this->setShowStreamlines(true);
	this->setStreamlinesAlpha(100);
	this->setStreamlinesComboMode(0);
    }
}

bool testPointOkay(const QImage &image, vec3 new_point, float d_sep, float d_test, float d_step)
{
   // cout << "test " << new_point[0] << ":" << new_point[1] << endl;

    //if point.x exceeds image dimensions, return false
    if (new_point[0] < 0.0f || new_point[0] > 1.0f)
	return false;

    //if point.y exceeds image dimensions, return false
    if (new_point[1] < 0.0f || new_point[1] > 1.0f)
	return false;

    int bounding_box_size_x_in_pixels = image.width() * d_sep * d_test * 2;
    int bounding_box_size_y_in_pixels = image.height() * d_sep * d_test * 2;

    int point_pos_x_in_pixels = new_point[0] * image.width();
    int point_pos_y_in_pixels = new_point[1] * image.height();

    for (int x_loop = point_pos_x_in_pixels - bounding_box_size_x_in_pixels / 2; x_loop < point_pos_x_in_pixels + bounding_box_size_x_in_pixels / 2; x_loop++)
	for (int y_loop = point_pos_y_in_pixels - bounding_box_size_y_in_pixels / 2; y_loop < point_pos_y_in_pixels + bounding_box_size_y_in_pixels / 2; y_loop++)
	{
	    if (x_loop < 0 || y_loop < 0)
		continue;

	    if (x_loop >= image.width() || y_loop >= image.height())
		continue;

	    if (sqrt(pow( (float) ((x_loop - point_pos_x_in_pixels) / image.width()), 2) +
		 pow( (float)((y_loop - point_pos_y_in_pixels) / image.height()), 2))
		>= d_sep * d_test)
		    continue;

		// if there has already a pixel been drawn
	    if (image.pixel(x_loop, y_loop) != QColor("black").rgb())
	    {
		//cout << "false " << x_loop << ":" << y_loop << endl;
		return false;
	    }
	}

    return true;
}

float testPointDistance(const QImage &image, vec3 new_point, float d_sep, float d_test, float d_step)
{
   // cout << "test " << new_point[0] << ":" << new_point[1] << endl;

    //if point.x exceeds image dimensions, return false
    if (new_point[0] < 0.0f || new_point[0] > 1.0f)
	return 1.0f;

    //if point.y exceeds image dimensions, return false
    if (new_point[1] < 0.0f || new_point[1] > 1.0f)
	return 1.0f;

    int bounding_box_size_x_in_pixels = image.width() * d_sep * 2;
    int bounding_box_size_y_in_pixels = image.height() * d_sep * 2;

    int point_pos_x_in_pixels = new_point[0] * image.width();
    int point_pos_y_in_pixels = new_point[1] * image.height();

    float min_distance = 1.0f;

    for (int x_loop = point_pos_x_in_pixels - bounding_box_size_x_in_pixels / 2; x_loop < point_pos_x_in_pixels + bounding_box_size_x_in_pixels / 2; x_loop++)
	for (int y_loop = point_pos_y_in_pixels - bounding_box_size_y_in_pixels / 2; y_loop < point_pos_y_in_pixels + bounding_box_size_y_in_pixels / 2; y_loop++)
	{
	    if (x_loop < 0 || y_loop < 0)
		continue;

	    if (x_loop >= image.width() || y_loop >= image.height())
		continue;

	    if (sqrt(pow((float)((x_loop - point_pos_x_in_pixels) / image.width()), 2) +
		 pow((float)((y_loop - point_pos_y_in_pixels) / image.height()), 2))
		>= d_sep)
		    continue;

		// if there has already a pixel been drawn
	    if (image.pixel(x_loop, y_loop) != QColor("black").rgb())
	    {
		float distance = sqrt(
			    pow((x_loop - point_pos_x_in_pixels) / (float)image.width(), 2) +
			    pow((y_loop - point_pos_y_in_pixels) / (float)image.height(), 2));

		if (distance < min_distance)
		    min_distance = distance;
	    }
	}

    return min_distance;
}

void FlowRenderer::updateStreamlinesTexture()
{
    checkInitialized();

    if (FlowData::instance()->isLoaded() == false)
	return;

    float d_sep = m_fStreamlines_dsep;
    float d_test = m_fStreamlines_dtest;
    float d_step = m_fStreamlines_dstep;
    int integration_mode = m_iStreamlinesMode;

    QImage image(m_iRenderWidth, m_iRenderHeight, QImage::Format_ARGB32);
    image.fill(QColor(0,0,0).rgb());

    // Painter erzeugen
    QPainter painter;
    painter.begin(&image);

    QPen pen;
    QBrush brush;

    pen.setColor(QColor(255,255,255));
    painter.setPen(pen);
    painter.setBrush(brush);


    QQueue<vec3>candidate_points;
    QQueue<QQueue <vec3> > all_streamlines;

    // set center of system and some others as candidates
    candidate_points.push_back(vec3(0.5, 0.5));
    /*for (double x=0.0; x < 1.0; x+= 0.1)
	for (double y=0.0; y < 1.0; y+= 0.1)
	candidate_points.push_back(vec3(x, y));*/


	while(candidate_points.size() > 0)
	{
		vec3 candidate = candidate_points.front(); // select first candidate in queue
		vec3 origin_candidate = candidate;  //save startpoint for going backward
		candidate_points.pop_front(); // and remove it from queue
		
		//cout << candidate[0] << ":" << candidate[1] << endl;

		if (testPointOkay(image, candidate, d_sep, d_test, d_step))
		{
			QQueue<vec3> streamline;
			bool streamline_running = true;

			streamline.push_back(candidate); // add candidate to streamline queue

			/* going forward */
			while(streamline_running && streamline.size() < m_iMaxStreamlineSegments)
			{
				// calculate direction
				float dir_at_candidate_x = FlowData::instance()->getChannel(0)->getValueNormPos( candidate[0], candidate[1] );
				float dir_at_candidate_y = FlowData::instance()->getChannel(1)->getValueNormPos( candidate[0], candidate[1] );
				vec3 direction = vec3(dir_at_candidate_x, dir_at_candidate_y);
				
				vec3 direction_left = vec3(-dir_at_candidate_y, dir_at_candidate_x);
				vec3 direction_right = vec3(dir_at_candidate_y, -dir_at_candidate_x);
				vec3 point_left = candidate + (direction_left / direction_left.length()) * d_sep;
				vec3 point_right = candidate + (direction_right / direction_right.length()) * d_sep;

				vec3 next_integration_point;

				if (integration_mode == 1){
					// do a runga kutta 2nd order integration step in time
					vec3 a = candidate + direction/direction.length() * d_step; // calculate euler step
					vec3 b = direction/direction.length() + (a/2.f) * d_step;	// do half a euler step
					next_integration_point = candidate + b * d_step;			// and use it at candidate point
				} else if (integration_mode == 2){
					// do a runga kutta 4th order integration step in time
					vec3 a = candidate + direction/direction.length() * d_step; // calculate euler step
					vec3 b = direction/direction.length() + (a/2.f) * d_step;	// do half a euler step
					vec3 c = direction/direction.length() + (b/2.f) * d_step;	// and again
					vec3 d = direction/direction.length() + c * d_step;			// and again
					next_integration_point = candidate + ((a + b*2.f + c*2.f + d)/6.f) * d_step; // compose
				} else {
					// go 1 euler integration step in time
					next_integration_point = candidate + (direction/direction.length()) * d_step;
				}

				if (testPointOkay(image, next_integration_point, d_sep, d_test, d_step))
				{
					candidate_points.push_back(point_left);
					candidate_points.push_back(point_right);

					if (m_bUseTapering)
					{
					    float thickness_coef;

					    float d = testPointDistance(image, next_integration_point, d_sep, d_test, d_step);

					    if (d > d_sep)
						thickness_coef = 1.0;
					    else
					    {

						thickness_coef = (d - d_test * d_sep) / (d_sep - d_test * d_sep);
					    }
					    if (thickness_coef < 0.0f)
						thickness_coef = 0.0f;
					    if (thickness_coef > 1.0f)
						thickness_coef = 1.0f;

					    //store thickness in vec3.z
					    next_integration_point[2] = thickness_coef;
					}

					streamline.push_back(next_integration_point);
					candidate = next_integration_point;
				}
				else
					streamline_running = false;
			} // end of streamline

			// draw streamline
			for (int i=1; i < streamline.size(); i++)
			{
				vec3 p1 = streamline.at(i-1);
				vec3 p2 = streamline.at(i);

				painter.drawLine(p1[0] * image.width(), p1[1] * image.height(), p2[0] * image.width(), p2[1] * image.height());
			}
			all_streamlines.push_back(streamline);
			streamline.clear();

			candidate = origin_candidate;
			streamline_running = true;

			/* ***************
			doing the same again but in the other direction
			******************/
			while(streamline_running && streamline.size() < m_iMaxStreamlineSegments)
			{
				// calculate direction
				float dir_at_candidate_x = -FlowData::instance()->getChannel(0)->getValueNormPos( candidate[0], candidate[1] );
				float dir_at_candidate_y = -FlowData::instance()->getChannel(1)->getValueNormPos( candidate[0], candidate[1] );
				vec3 direction = vec3(dir_at_candidate_x, dir_at_candidate_y);

				vec3 direction_left = vec3(-dir_at_candidate_y, dir_at_candidate_x);
				vec3 direction_right = vec3(dir_at_candidate_y, -dir_at_candidate_x);
				vec3 point_left = candidate + (direction_left / direction_left.length()) * d_sep;
				vec3 point_right = candidate + (direction_right / direction_right.length()) * d_sep;

				vec3 next_integration_point;

				if (integration_mode == 1){
					// do a runga kutta 2nd order integration step in time
					vec3 a = candidate + direction/direction.length() * d_step; // calculate euler step
					vec3 b = direction/direction.length() + (a/2.f) * d_step;	// do half a euler step
					next_integration_point = candidate + b * d_step;			// and use it at candidate point
				} else if (integration_mode == 2){
					// do a runga kutta 4th order integration step in time
					vec3 a = candidate + direction/direction.length() * d_step; // calculate euler step
					vec3 b = direction/direction.length() + (a/2.f) * d_step;	// do half a euler step
					vec3 c = direction/direction.length() + (b/2.f) * d_step;	// and again
					vec3 d = direction/direction.length() + c * d_step;			// and again
					next_integration_point = candidate + ((a + b*2.f + c*2.f + d)/6.f) * d_step; // compose
				} else {
					// go 1 euler integration step in time
					next_integration_point = candidate + (direction/direction.length()) * d_step;
				}

				if (testPointOkay(image, next_integration_point, d_sep, d_test, d_step))
				{
					candidate_points.push_back(point_left);
					candidate_points.push_back(point_right);

					if (m_bUseTapering)
					{
					    float thickness_coef;

					    float d = testPointDistance(image, next_integration_point, d_sep, d_test, d_step);

					    if (d > d_sep)
						thickness_coef = 1.0;
					    else
					    {

						thickness_coef = (d - d_test * d_sep) / (d_sep - d_test * d_sep);
					    }
					    if (thickness_coef < 0.0f)
						thickness_coef = 0.0f;
					    if (thickness_coef > 1.0f)
						thickness_coef = 1.0f;

					    //store thickness in vec3.z
					    next_integration_point[2] = thickness_coef;
					}

					streamline.push_back(next_integration_point);
					candidate = next_integration_point;
				}
				else
					streamline_running = false;
			} // end of streamline

			// draw streamline
			for (int i=1; i < streamline.size(); i++)
			{
				vec3 p1 = streamline.at(i-1);
				vec3 p2 = streamline.at(i);

				painter.drawLine(p1[0] * image.width(), p1[1] * image.height(), p2[0] * image.width(), p2[1] * image.height());
			}

			all_streamlines.push_back(streamline);

		}

	}

	QImage image_new(m_iRenderWidth, m_iRenderHeight, QImage::Format_ARGB32);
	QImage glyph("pfeil.jpg");

	image_new.fill(QColor(0,0,0).rgb());
	QPainter painter2;

	// Painter erzeugen
	painter2.begin(&image_new);

	pen.setColor(QColor(255,255,255));
	painter2.setPen(pen);
	painter2.setBrush(brush);
	painter2.setRenderHint(QPainter::Antialiasing, true);
	pen.setCapStyle(Qt::RoundCap);

	for (int l=0; l < all_streamlines.size(); l++)
	{
	    QQueue<vec3> current_streamline = all_streamlines.at(l);
	    for (int j=1; j < current_streamline.size(); j++)
	    {
		    vec3 p1 = current_streamline.at(j-1);
		    vec3 p2 = current_streamline.at(j);

			// apply texturing technique from paper
			int streamline_tex_mode = m_iStreamlineTexturingMode;
			int N = m_iStreamlineTexturingPeriodLength;
			int streamline_tex_color = 255; // set white as initial state

			if (streamline_tex_mode==0)
			    streamline_tex_color = 255.0f;
			else
			if (streamline_tex_mode==1)
			    streamline_tex_color = 255.f * 0.5f * (1 + sin( (2*M_PI*j) / (float)N ) ); // use formula f1 for texturing
			else
			if (streamline_tex_mode==2)
			    streamline_tex_color = (int) (255.f * ( (j % N) / (float)(N-1) ) ); // use formula f2 for texturing

		    pen.setColor(QColor(streamline_tex_color, streamline_tex_color, streamline_tex_color));

		    if (m_bUseTapering)
		    {
			float thickness_coef1 = p1[2];
			float thickness_coef2 = p2[2];

			int interpolation_steps = 10;
			float dir_x = (p2[0] - p1[0]) / interpolation_steps;
			float dir_y = (p2[1] - p1[1]) / interpolation_steps;

			if (thickness_coef1 != thickness_coef2) //we must interpolate
			for (int i=0; i < interpolation_steps; i++)
			{
			    pen.setWidth(m_iStreamlineWidth * (thickness_coef1 + i * ((thickness_coef2 - thickness_coef1)/ interpolation_steps)));
			    painter2.setPen(pen);

			    painter2.drawLine((p1[0] + dir_x * i) * image.width(), (p1[1] + dir_y * i) * image.height(), (p1[0] + dir_x * (i+1)) * image.width(), (p1[1] + dir_y * (i+1)) * image.height());

			}
			else
			{
			    pen.setWidth(m_iStreamlineWidth * thickness_coef1);
			    painter2.setPen(pen);

			    painter2.drawLine(p1[0] * image.width(), p1[1] * image.height(), p2[0] * image.width(), p2[1] * image.height());
			}
		    }
		    else
		    {
			pen.setWidth(m_iStreamlineWidth);

			painter2.setPen(pen);

			painter2.drawLine(p1[0] * image.width(), p1[1] * image.height(), p2[0] * image.width(), p2[1] * image.height());

			pen.setWidth(1.0);
			painter2.setPen(pen);
		    }


	    }
	}

	/* generate glyph texture */
	/* ************************************ */

	static QGLFramebufferObject fbo(1024, 1024);

	QGLShaderProgram shader;
	shader.addShaderFromSourceFile(QGLShader::Vertex, "shaders/glyph.vs");
	shader.addShaderFromSourceFile(QGLShader::Fragment, "shaders/glyph.fs");
	shader.link();
	shader.bind();

	QImage gl_glyph_image = QGLWidget::convertToGLFormat(glyph);

	glEnable(GL_TEXTURE_2D);
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, m_gliGlyphTexture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, gl_glyph_image.width(), gl_glyph_image.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, gl_glyph_image.bits());
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_CLAMP);

	shader.setUniformValue("glyph", 0);
	fbo.bind();

	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glLoadIdentity();							// Reset The Projection Matrix
	glViewport(0, 0, 1024, 1024);
	glMatrixMode(GL_MODELVIEW);						// Select The Modelview Matrix
	glLoadIdentity();
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	for (int l=0; l < all_streamlines.size(); l++)
	{
	    QQueue<vec3> current_streamline = all_streamlines.at(l);
	    for (int j=1; j < current_streamline.size(); j++)
	    {
		    vec3 p1 = current_streamline.at(j-1);
		    vec3 p2 = current_streamline.at(j);

		    double scale = m_fStreamlines_dstep;

		    p1[2] = 0.0;
		    p2[2] = 0.0;

		    vec3 direction_normalized = (p2 - p1) / (p2 - p1).length();
		    vec3 p2_real = p1 + (direction_normalized) * scale;
		    vec3 p3_real = p2 + vec3(direction_normalized[1], -direction_normalized[0]) * scale * 0.44;
		    vec3 p4_real = p1 + vec3(direction_normalized[1], -direction_normalized[0]) * scale * 0.44;

		    glBegin(GL_QUADS);
		    glMultiTexCoord2d(GL_TEXTURE0, 0, 0);
		    glVertex3f(p1[0] * 2.0 - 1.0, p1[1] * 2.0 - 1.0, 0.0f);
		    glMultiTexCoord2d(GL_TEXTURE0, 0, 1);
		    glVertex3f(p2_real[0] * 2.0 - 1.0, p2_real[1] * 2.0 - 1.0, 0.0f);
		    glMultiTexCoord2d(GL_TEXTURE0, 1, 1);
		    glVertex3f(p3_real[0] * 2.0 - 1.0, p3_real[1] * 2.0 - 1.0, 0.0f);
		    glMultiTexCoord2d(GL_TEXTURE0, 1, 0);
		    glVertex3f(p4_real[0] * 2.0 - 1.0, p4_real[1] * 2.0 - 1.0, 0.0f);
		    glEnd();

	    }
	}

	shader.release();
	fbo.release();
	m_gliStreamlinesGlyphsTexture = fbo.texture();

	/* finished glyph texture */

	/* convert createt QImage to opengl texture (lines only, glyphs are already an opengl texture) */

	QImage gl_image = QGLWidget::convertToGLFormat(image_new);

	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, m_gliStreamlinesTexture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, gl_image.width(), gl_image.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, gl_image.bits());
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_CLAMP);
	glBindTexture(GL_TEXTURE_2D, 0);

}

void FlowRenderer::updateArrowTexture()
{
    checkInitialized();

    if (FlowData::instance()->isLoaded() == false)
	return;

    //create image where we can draw into
    QImage image(m_iRenderWidth, m_iRenderHeight, QImage::Format_ARGB32);
    image.fill(QColor(0,0,0,0).rgba());

    // Painter erzeugen
    QPainter painter;
    painter.begin(&image);

    QPen pen;
    QBrush brush;

    painter.setPen(pen);
    painter.setBrush(brush);

    FlowData *flow_instance = FlowData::instance();

    int x, y;
    for (x=0; x < image.width(); x+= 10)
	for (y=0; y < image.height(); y+= 10)
	{
	    pen.setColor(QColor(255,255,255,255));

	    int scale = 5;
	    float x_direction = flow_instance->getChannel(0)->getValueNormPos((float)x / image.width(), (float)y / image.height());
	    float y_direction = flow_instance->getChannel(1)->getValueNormPos((float)x / image.width(), (float)y / image.height());
	    float vector_length = sqrt(x_direction * x_direction + y_direction * y_direction);
	    x_direction /= vector_length;
	    y_direction /= vector_length;


	    painter.setPen(pen);
	    painter.drawPoint(x,y);
	    painter.drawLine(x, y, x+x_direction * scale, y+y_direction * scale);
	}


    //convert image to opengl texture
    QImage gl_image = QGLWidget::convertToGLFormat(image);

    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, m_gliArrowTexture);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, gl_image.width(), gl_image.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, gl_image.bits());
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_CLAMP);
    glBindTexture(GL_TEXTURE_2D, 0);

}


void FlowRenderer::render()
{
    checkInitialized();

    if (FlowData::instance()->isLoaded() == false)
	return;

    glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
    glLoadIdentity();							// Reset The Projection Matrix
    glViewport(0, 0, m_iRenderWidth, m_iRenderHeight);
    glMatrixMode(GL_MODELVIEW);						// Select The Modelview Matrix
    glLoadIdentity();
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    m_ShaderProgram.bind();

    glActiveTexture(GL_TEXTURE0);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, FlowData::instance()->getChannel(0)->get2DTexture());
    glActiveTexture(GL_TEXTURE1);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, FlowData::instance()->getChannel(1)->get2DTexture());
    glActiveTexture(GL_TEXTURE2);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, FlowData::instance()->getChannel(3)->get2DTexture());
    glActiveTexture(GL_TEXTURE3);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, FlowData::instance()->getChannel(4)->get2DTexture());

    glActiveTexture(GL_TEXTURE4);
    glEnable(GL_TEXTURE_1D);
    glBindTexture(GL_TEXTURE_1D, TransferFunction::instance(0)->getTextureName());
    glActiveTexture(GL_TEXTURE5);
    glEnable(GL_TEXTURE_1D);
    glBindTexture(GL_TEXTURE_1D, TransferFunction::instance(1)->getTextureName());

    glActiveTexture(GL_TEXTURE6);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, m_gliArrowTexture);
    glActiveTexture(GL_TEXTURE7);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, m_gliStreamlinesTexture);
    glActiveTexture(GL_TEXTURE8);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, m_gliStreamlinesGlyphsTexture);


    m_ShaderProgram.setUniformValue("channel0", 0);
    m_ShaderProgram.setUniformValue("channel1", 1);
    m_ShaderProgram.setUniformValue("channel3", 2);
    m_ShaderProgram.setUniformValue("channel4", 3);
    m_ShaderProgram.setUniformValue("transferfunction1", 4);
    m_ShaderProgram.setUniformValue("transferfunction2", 5);
    m_ShaderProgram.setUniformValue("arrows", 6);
    m_ShaderProgram.setUniformValue("streamlines", 7);
    m_ShaderProgram.setUniformValue("streamlines_glyphs", 8);
    m_ShaderProgram.setUniformValue("show_arrows", m_bShowArrowTexture);
    m_ShaderProgram.setUniformValue("show_streamlines", m_bShowStreamlinesTexture);
    m_ShaderProgram.setUniformValue("show_glyphs", m_bUseGlyphs);
    m_ShaderProgram.setUniformValue("colorcoding", m_iColorCoding);
    m_ShaderProgram.setUniformValue("add_streamlines", m_bAddStreamlines);
    m_ShaderProgram.setUniformValue("alpha_streamlines", m_fStreamlines_Alpha);

    static int loop=0;
    loop++;

    /*if (loop % 2 == 0)
    glBegin(GL_TRIANGLE_STRIP);
    glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
    glVertex3f(-1.0f, -1.0f, 0.0f);  //left bottom
    glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 1.0f);
    glVertex3f(-1.0f, 1.0f, 0.0f);  //left top
    glMultiTexCoord2f(GL_TEXTURE0, 1.0f, 0.0f);
    glVertex3f(1.0f, -1.0f, 0.0f);  //right bottom
    glMultiTexCoord2f(GL_TEXTURE0, 1.0f, 1.0f);
    glVertex3f(1.0f, 1.0f, 0.0f);  //right top
    glEnd();*/

    FlowGeometry *geometry = FlowData::instance()->getGeometry();
    int dimension_x = geometry->getDimY();
    int dimension_y = geometry->getDimX();

    float max_x = geometry->getPosY(dimension_y-1);
    float max_y = geometry->getPosX(dimension_y * dimension_x - 1);

    float *x_positions = new float[dimension_x];
    float *y_positions = new float[dimension_y];

    for (int x=0; x < dimension_x; x++)
	x_positions[x] = geometry->getPosY(x) / max_x;

    for (int y=0; y < dimension_y; y++)
	y_positions[y] = geometry->getPosX(y*dimension_y) / max_y;

    for (int x=0; x < dimension_x-1; x++)
    {
	glBegin(GL_TRIANGLE_STRIP);
	for (int y=0; y < dimension_y; y++)
	{
	    glMultiTexCoord2f(GL_TEXTURE0, (float) x / dimension_x, (float) y / dimension_y);
	    glMultiTexCoord2f(GL_TEXTURE1, (float) x_positions[x], (float) y_positions[y]);
	    glVertex3f(x_positions[x] * 2.0 - 1.0, y_positions[y] * 2.0 - 1.0, 0.0f);
	    glMultiTexCoord2f(GL_TEXTURE0, (float) (x+1) / dimension_x, (float) y / dimension_y);
	    glMultiTexCoord2f(GL_TEXTURE1, (float) x_positions[x+1], (float) y_positions[y]);
	    glVertex3f(x_positions[x+1] * 2.0 - 1.0, y_positions[y] * 2.0 - 1.0, 0.0f);
	}
	glEnd();
    }

    delete[] x_positions;
    delete[] y_positions;

    m_ShaderProgram.release();
}


