#include "VolumeData.h"
#include "Logger.h"
#include <QFile>
#include <exception>
#include <iostream>
#include <cmath>

using namespace std;

VolumeData singelton_instance_volumedata;

/*!
* \brief This class produces a singleton instance and this method returns it.
* \return pointer to singleton instance.
*/
VolumeData* VolumeData::instance()
{
	return &singelton_instance_volumedata;
}

VolumeData::VolumeData() : QObject(0)
{
	this->data = NULL;
	this->textureName = 0;
	this->gradientTextureName = 0;
	this->m_iDepth = this->m_iHeight = this->m_iWidth = 100;
}

VolumeData::~VolumeData(void)
{
	if (this->data != NULL)
		delete [] this->data;
}

/*!
* \brief This method loads a volumedata file and prepares the data for
* usage with OpenGL.
* \param strFilename The Filename or Filepath to load
*
* A 3D Texture storing the volumedata and another 3D Texture
* storing the pre-computed gradient information gets created.
* Also the historgram data is created while parsing the file.
*
* Die Dateien bestehen aus einem 6 Byte Header gefolgt von den
* eigentlichen Daten. Die ersten zwei Byte des Headers geben an
* wie gro das Volumen in x-Richtung ist, die folgenden zwei Byte
* geben an wie gro es in y-Richtung ist und die nchsten zwei Byte
* geben die Gre der z-Dimension an.
* Die eigentlichen Daten sind als 16 Bit pro Datenwert gespeichert wobei nur 12 Bit in Verwendung sind.
*
* Parts of this method have been taken from the VisLU students framework and rewritten for Qt.
*/
void VolumeData::loadDataSet(QString strFilename)
{
	Logger::instance()->log(Logger::Debug, "Opening File");
	QFile file(strFilename);

	if (!file.open(QIODevice::ReadOnly)){
		Logger::instance()->log(Logger::Error, "error opening the file !!!");
		return;
	}

	// read the header
	Logger::instance()->log(Logger::Debug, "Reading file header");

	unsigned short uWidth, uHeight, uDepth;

	if (file.read((char *)&uWidth, sizeof(unsigned short)) != sizeof(unsigned short))
		Logger::instance()->log(Logger::Error, "Failed to read width");

	if (file.read((char *)&uHeight, sizeof(unsigned short)) != sizeof(unsigned short))
		Logger::instance()->log(Logger::Error, "Failed to read height");

	if (file.read((char *)&uDepth, sizeof(unsigned short)) != sizeof(unsigned short))
		Logger::instance()->log(Logger::Error, "Failed to read depth");

	m_iWidth = (int) uWidth;
	m_iHeight = (int) uHeight;
	m_iDepth = (int) uDepth;

	// QString logger_message = "Data dimensions: x=" + QString::number(m_iWidth) + QString(", y=") + QString::number(m_iHeight) + QString(", z=") + QString::number(m_iDepth);
	// Logger::instance()->log(Logger::Debug, logger_message.toStdString());

	if (this->data == NULL)
		delete[] this->data;
	if (this->histodata == NULL)
		delete[] this->histodata;

	Logger::instance()->log(Logger::Debug, "creating data array");

	this->data = new GLfloat[uWidth*uHeight*uDepth];
	this->histodata = new int[4096];

	// initialize histodata to zero
	for (int i=0; i < 4096; i++)
		this->histodata[i] = 0;

	Logger::instance()->log(Logger::Debug, "Iterating over data");
	for (int i=0; i < uWidth*uHeight*uDepth; i++)
	{
		unsigned short puffer;

		/* int z = i / (uWidth * uHeight);
		int y = (i - z*uWidth*uHeight) / (uWidth);
		int x = i % (uWidth);*/

		file.read((char *) &puffer, sizeof(unsigned short));
		// create the histogram
		this->histodata[int(puffer)]++;
		// fill the data array and normalize
		this->data[i] = min(1.0f, float(puffer) / 4096.0f);
	}

	Logger::instance()->log(Logger::Debug, "Closing File");
	file.close();
	Logger::instance()->log(Logger::Debug, "File loaded successfully");

	/************************************************************************/
	/* Create the 3D Texture                                                */
	/************************************************************************/
	if (this->textureName==0)
		glGenTextures(1, &this->textureName);

	glEnable(GL_TEXTURE_3D);
	glBindTexture(GL_TEXTURE_3D, this->textureName);

	glTexImage3D(GL_TEXTURE_3D, 0, GL_LUMINANCE , uWidth,uHeight,uDepth, 0, GL_LUMINANCE, GL_FLOAT, this->data);

	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_R, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_T, GL_CLAMP);

	/************************************************************************/
	/* Calculate Gradient and save as well in a 3D Texture                  */
	/************************************************************************/
	// get enough space for the gradients
	gradients = new float[3 * m_iWidth * m_iHeight * m_iDepth]; // x*y*z*3 (vec3f at every pos)
	float *gradient = new float[3];
	
	// indices
	int actIndex;
	int preIndexX, postIndexX;
	int preIndexY, postIndexY;
	int preIndexZ, postIndexZ;
	int gradientsIndex;
	const int iSlice = m_iWidth * m_iHeight;

	// iterate over VolumeData
	for (int k = 0, kThird = 0; kThird < m_iDepth; k = k + 3, kThird++)
	{
		for (int j = 0, jThird = 0; jThird < m_iHeight; j = j + 3, jThird++)
		{
			for (int i = 0, iThird = 0; iThird < m_iWidth; i = i + 3, iThird++)
			{
				// calculate the necessary indices
				gradientsIndex = i + j * m_iWidth + k * iSlice;
				actIndex = iThird + jThird * m_iWidth + kThird * iSlice;
				preIndexX = actIndex - 1;
				postIndexX = actIndex + 1;
				preIndexY = actIndex - m_iWidth;
				postIndexY = actIndex + m_iWidth;
				preIndexZ = actIndex - iSlice;
				postIndexZ = actIndex + iSlice;

				// X-Direction
				if (iThird == 0) {
					// left border
					gradient[0] = this->data[postIndexX] - this->data[actIndex];
				} else if (iThird == m_iWidth - 1) {
					// right border
					gradient[0] = this->data[actIndex] - this->data[preIndexX];
				} else {
					gradient[0] = 0.5f * (this->data[postIndexX] - this->data[preIndexX]);
				}
				// Y-Direction
				if (jThird == 0) {
					// bottom border
					gradient[1] = this->data[postIndexY] - this->data[actIndex];
				} else if (jThird == m_iHeight - 1) {
					// top border
					gradient[1] = this->data[actIndex] - this->data[preIndexY];
				} else {
					gradient[1] = 0.5f * (this->data[postIndexY] - this->data[preIndexY]);
				}
				// Z-Direction
				if (kThird == 0) {
					// front border
					gradient[2] = this->data[postIndexZ] - this->data[actIndex];
				} else if (kThird == m_iDepth - 1) {
					// back border
					gradient[2] = this->data[actIndex] - this->data[preIndexZ];
				} else {
					gradient[2] = 0.5f * (this->data[postIndexZ] - this->data[preIndexZ]);
				}

				// save and shift gradients from [-1...+1] to [0...1] range
				gradients[gradientsIndex + 0] = (gradient[0] + 1.0f) / 2.0f;
				gradients[gradientsIndex + 1] = (gradient[1] + 1.0f) / 2.0f;
				gradients[gradientsIndex + 2] = (gradient[2] + 1.0f) / 2.0f;
			}
		}
		// string message = "\r- Computing gradients (" + (kThird * 100) / (m_iDepth-1) + "%) ...";
		// Logger::instance()->log(Logger::Debug, message);
	}

	if (this->gradientTextureName==0)
		glGenTextures(1, &this->gradientTextureName);

	glEnable(GL_TEXTURE_3D);
	glBindTexture(GL_TEXTURE_3D, this->gradientTextureName);

	glTexImage3D(GL_TEXTURE_3D, 0, GL_RGB , uWidth,uHeight,uDepth, 0, GL_RGB, GL_FLOAT, this->gradients);

	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_R, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_T, GL_CLAMP);
}

/*!
* \brief This method returns the pointer to the volumetric data.
* \return The pointer to the float array storing the volumetric data.
*/
GLfloat* VolumeData::getVolumetricData()
{
	if (this->data == NULL)
		Logger::instance()->log(Logger::Fatal, "No VolumeData set (this->data == NULL) but getVolumetricData() called");

	return this->data;
}

/*!
* \brief This method returns the pointer to the histogram data.
* \return The pointer to the int array storing the histogram data.
*/
int* VolumeData::getHistogramData()
{
	// if (this->histodata == NULL)
	//	Logger::instance()->log(Logger::Fatal, "No VolumeData set (this->histodata == NULL) but getHistogramData() called");

	return this->histodata;
}

/*!
* \brief This method returns the pointer to the normalized histogram data.
* \return The pointer to the int array storing the normalized histogram data.
*/
float* VolumeData::getNormalizedHistogramData()
{
	if (histodata == NULL)
		return NULL;

	// find the maximum
	static int max=0;
	for (int i=0; i < 4096; i++){
		if (this->histodata[i] > max){
			max = this->histodata[i];
		}
	}

	// new target
	static float *data = new float[4096];

	// normalize
	for (int i=0; i < 4096; i++)
		data[i] = this->histodata[i] / (float) max;

	return data;
}

/*!
* \brief This method returns the pointer to the logarithmic scaled normalized histogram data.
* \return The pointer to the int array storing the logarithmic scaled normalized histogram data.
*/
float* VolumeData::getLogNormalizedHistogramData(){
	if (histodata == NULL)
		return NULL;

	// new target
	static float *data = new float[4096];

	// apply logarithm on histodata
	for (int i=0; i < 4096; i++)
	{
	    if (this->histodata[i] > 0)
		data[i] = log( (float) (this->histodata[i]) );
	    else
		data[i] = 0;
	}

	// find the maximum
	static int max2=0;
	for (int i=0; i < 4096; i++)
		if (data[i] > max2)
			max2 = data[i];

	// normalize
	for (int i=0; i < 4096; i++)
		data[i] = data[i] / (float) max2;

	return data;
}

/*!
* \brief This method returns the OpenGL identifier for the 3D texture storing the volumetric data.
* \return The OpenGL identifier for the 3D texture storing the volumetric data.
*/
GLuint VolumeData::getTextureName()
{
	// if (this->textureName == 0)
	// Logger::instance()->log(Logger::Error, "3D Volume Texture not created yet");

	return this->textureName;
}

/*!
* \brief This method returns the OpenGL identifier for the 3D texture storing the precomputed gradients.
* \return The OpenGL identifier for the 3D texture storing the precomputed gradients.
*/
GLuint VolumeData::getGradientTextureName()
{
	return this->gradientTextureName;
}
