#include <GL/glew.h>
#include <QtOpenGL>
#include <iostream>

#include "RaytracedRenderer.h"
#include "TransferFunction.h"
#include "VolumeData.h"
#include "Visualizer.h"
#include "Logger.h"

RaytracedRenderer instance_raytraced_renderer;

RaytracedRenderer *RaytracedRenderer::instance()
{
    return &instance_raytraced_renderer;
}

RaytracedRenderer::RaytracedRenderer() : m_bInitialized(false)
{

}

void RaytracedRenderer::initialize()
{
	if (!this->m_bInitialized)
	{
    //load shaders or whatever we need to load after construction (because at construction-time, we dont have a valid opengl context!)
	this->m_pBackSidesFBO = new QGLFramebufferObject(400, 400);
	this->m_pFrontSidesFBO = new QGLFramebufferObject(400, 400);
	this->m_pRayCastFBO = new QGLFramebufferObject(400,400);
	this->m_iHeight = 400;
	this->m_iWidth = 400;
	m_fFBOSizeFactor = 0.5f;
	m_fClipXLeft = -1.0f;
	m_fClipXRight = 1.0f;
	m_fClipYTop = 1.0f;
	m_fClipYBottom = -1.0f;
	m_fClipZBack = 1.0f;
	m_fClipZFront = -1.0f;

	m_bAllowFixedLowSteps = false;
	m_bFixedLowSteps = false;
	m_bProgressiveRefinement = true;
	m_bShadingEnabled = true;

	QGLShader shadervs(QGLShader::Vertex);
	QGLShader shaderfs(QGLShader::Fragment);
	m_qPosIsColorShader.setParent(this);
	shadervs.compileSourceFile(QString("shaders/position_is_color.vs"));
	shaderfs.compileSourceFile(QString("shaders/position_is_color.fs"));
	m_qPosIsColorShader.addShader(&shadervs);
	m_qPosIsColorShader.addShader(&shaderfs);
	m_qPosIsColorShader.link();

	m_qRaytraceShader.setParent(this);
	shadervs.compileSourceFile(QString("shaders/raytrace.vs"));
	shaderfs.compileSourceFile(QString("shaders/raytrace.fs"));
	m_qRaytraceShader.addShader(&shadervs);
	m_qRaytraceShader.addShader(&shaderfs);
	m_qRaytraceShader.link();

	m_qDrawResultShader.setParent(this);
	shadervs.compileSourceFile(QString("shaders/drawresult.vs"));
	shaderfs.compileSourceFile(QString("shaders/drawresult.fs"));
	m_qDrawResultShader.addShader(&shadervs);
	m_qDrawResultShader.addShader(&shaderfs);
	m_qDrawResultShader.link();

	m_bInitialized = true;
    }
}

void RaytracedRenderer::resize(int width, int height)
{
    if (m_bInitialized == false)
	initialize();

    delete this->m_pBackSidesFBO;
    delete this->m_pFrontSidesFBO;

    this->m_pBackSidesFBO = new QGLFramebufferObject(width * m_fFBOSizeFactor, height * m_fFBOSizeFactor);
    this->m_pFrontSidesFBO = new QGLFramebufferObject(width * m_fFBOSizeFactor, height * m_fFBOSizeFactor);
    this->m_pRayCastFBO = new QGLFramebufferObject(width * m_fFBOSizeFactor, height * m_fFBOSizeFactor);

    this->m_iHeight = height;
    this->m_iWidth = width;
}

void RaytracedRenderer::setClippingValues(float fXLeft, float fXRight, float fYTop, float fYBottom, float fZBack, float fZFront)
{
    if (fXLeft==fXRight)
	fXLeft-=0.01f;

    if (fYTop==fYBottom)
	fYBottom-=0.01f;

    if (fZBack==fZFront)
	fZBack-=0.01f;

    m_fClipXLeft = fXLeft;
    m_fClipXRight = fXRight;
    m_fClipYTop = fYTop;
    m_fClipYBottom = fYBottom;
    m_fClipZBack = fZBack;
    m_fClipZFront = fZFront;
}

void RaytracedRenderer::itlDrawColorCube()
{
	const float clip_x_left = m_fClipXLeft;
	const float clip_x_right = m_fClipXRight;
	const float clip_y_top = m_fClipYTop;
	const float clip_y_bottom = m_fClipYBottom;
	const float clip_z_front = m_fClipZBack;
	const float clip_z_back = m_fClipZFront;


	glPushAttrib(GL_ALL_ATTRIB_BITS);
	glPushMatrix();

	glDisable(GL_TEXTURE_1D);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_3D);

	m_qPosIsColorShader.bind();

	glBegin(GL_QUADS);

	//left side
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_back);
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_front);
	glVertex3f(clip_x_left, clip_y_top, clip_z_front);
	glVertex3f(clip_x_left, clip_y_top, clip_z_back);

	//front side
	glVertex3f(clip_x_left, clip_y_top, clip_z_front);
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_front);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_front);
	glVertex3f(clip_x_right, clip_y_top, clip_z_front);

	//right side
	glVertex3f(clip_x_right, clip_y_top, clip_z_back);
	glVertex3f(clip_x_right, clip_y_top, clip_z_front);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_front);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_back);

	//back side
	glVertex3f(clip_x_right, clip_y_top, clip_z_back);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_back);
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_back);
	glVertex3f(clip_x_left, clip_y_top, clip_z_back);

	//bottom
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_back);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_back);
	glVertex3f(clip_x_right, clip_y_bottom, clip_z_front);
	glVertex3f(clip_x_left, clip_y_bottom, clip_z_front);

	//top
	glVertex3f(clip_x_left, clip_y_top, clip_z_back);
	glVertex3f(clip_x_left, clip_y_top, clip_z_front);
	glVertex3f(clip_x_right, clip_y_top, clip_z_front);
	glVertex3f(clip_x_right, clip_y_top, clip_z_back);

	glEnd();

	m_qPosIsColorShader.release();

	glPopMatrix();
	glPopAttrib();
}

void RaytracedRenderer::preRender()
{
	glPushAttrib(GL_ALL_ATTRIB_BITS);
	glPushMatrix();

	glEnable(GL_CULL_FACE);
	glCullFace(GL_FRONT);

	this->m_pBackSidesFBO->bind();
	glViewport(0,0,m_iWidth * m_fFBOSizeFactor,m_iHeight * m_fFBOSizeFactor);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	this->itlDrawColorCube();
	this->m_pBackSidesFBO->release();

	glCullFace(GL_BACK);

	this->m_pFrontSidesFBO->bind();
	glViewport(0,0,m_iWidth * m_fFBOSizeFactor,m_iHeight * m_fFBOSizeFactor);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	this->itlDrawColorCube();
	this->m_pFrontSidesFBO->release();

	glPopMatrix();
	glPopAttrib();
}

void RaytracedRenderer::render()
{
	glPushAttrib(GL_ALL_ATTRIB_BITS);
	glPushMatrix();
	this->m_pRayCastFBO->bind();
	glViewport(0,0,m_iWidth * m_fFBOSizeFactor,m_iHeight * m_fFBOSizeFactor);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glDisable(GL_TEXTURE_1D);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_3D);

	m_qRaytraceShader.bind();
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("mode")), m_iMode);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("height")), VolumeData::instance()->getHeight());
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("width")), VolumeData::instance()->getWidth());
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("depth")), VolumeData::instance()->getDepth());
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("shading_enabled")), m_bShadingEnabled);
	if (m_bFixedLowSteps && m_bAllowFixedLowSteps)
	    m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("steps_mode")), 50);
	else
	    m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("steps_mode")), m_iSteps * 100);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_1D, TransferFunction::instance()->getTextureName());
	glEnable(GL_TEXTURE_1D);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("transferfunction")), 0);

	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_3D,VolumeData::instance()->getTextureName());
	glEnable(GL_TEXTURE_3D);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("volumetexture")), 1);

	glActiveTexture(GL_TEXTURE2);
	glBindTexture(GL_TEXTURE_3D,VolumeData::instance()->getGradientTextureName());
	glEnable(GL_TEXTURE_3D);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("gradients")), 2);

	glActiveTexture(GL_TEXTURE3);
	glBindTexture(GL_TEXTURE_2D, this->m_pBackSidesFBO->texture());
	glEnable(GL_TEXTURE_2D);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("backface_fbo")), 3);

	glActiveTexture(GL_TEXTURE4);
	glBindTexture(GL_TEXTURE_2D, this->m_pFrontSidesFBO->texture());
	glEnable(GL_TEXTURE_2D);
	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("frontface_fbo")), 4);

	glBegin(GL_QUADS);
		glColor3f(1.0f,1.0f,1.0f);
		glTexCoord2f(0.0f, 0.0f);
		glVertex2f(-1.0f, -1.0f);
		glTexCoord2f(1.0f, 0.0f);
		glVertex2i(1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f);
		glVertex2i(1.0f, 1.0f);
		glTexCoord2f(0.0f ,1.0f);
		glVertex2i(-1.0f, 1.0f);
	glEnd();

	m_qRaytraceShader.release();

	this->m_pRayCastFBO->release();

	glPopMatrix();
	glPopAttrib();


	/* now draw the rendered texture */

	glPushAttrib(GL_ALL_ATTRIB_BITS);
	glPushMatrix();

	m_qDrawResultShader.bind();

	glDisable(GL_TEXTURE_1D);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_3D);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, this->m_pRayCastFBO->texture());
	glEnable(GL_TEXTURE_2D);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	m_qRaytraceShader.setUniformValue(m_qRaytraceShader.uniformLocation(QString("raycast_fbo")), 0);

	glBegin(GL_QUADS);
		glColor3f(1.0f,1.0f,1.0f);
		glTexCoord2f(0.0f, 0.0f);
		glVertex2f(-1.0f, -1.0f);
		glTexCoord2f(1.0f, 0.0f);
		glVertex2i(1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f);
		glVertex2i(1.0f, 1.0f);
		glTexCoord2f(0.0f ,1.0f);
		glVertex2i(-1.0f, 1.0f);
	glEnd();

	m_qDrawResultShader.release();

	glPopMatrix();
	glPopAttrib();
}

void RaytracedRenderer::setFixedLowSteps()
{
    m_bFixedLowSteps = true;

    if (m_bProgressiveRefinement)
    {
	this->m_fFBOSizeFactor /= 4;
	this->resize(m_iWidth, m_iHeight);
    }
}

void RaytracedRenderer::unsetFixedLowSteps()
{
    m_bFixedLowSteps = false;

    if (m_bProgressiveRefinement)
    {
	this->m_fFBOSizeFactor *= 4;
	this->resize(m_iWidth, m_iHeight);
    }
}

void RaytracedRenderer::setFBOSize(int iSelected)
{
    //iSelected is the index of the selected line
    //0 means factor 1/8
    //1 means factor 1/4
    //2 means factor 1/2
    //3 means factor 1
    //4 means factor 2
    //and so on

    this->m_fFBOSizeFactor = pow(2.0f,(float)(iSelected-3));
    this->resize(m_iWidth, m_iHeight);
}

void RaytracedRenderer::saveRenderedImageToFile(QString filename)
{
    //First, store old values
    float old_fFBOSizeFactor = m_fFBOSizeFactor;
    int old_iSteps = m_iSteps;

    //Then set values to generate image with high resolution and high precision
    m_fFBOSizeFactor = 2;
    m_iSteps = 7;
    this->resize(m_iWidth, m_iHeight);

    //render image
    Visualizer::instance()->render();

    //get image from fbo and save to file
    QImage image = m_pRayCastFBO->toImage();
    image.save(filename);

    //restora old values
    m_iSteps = old_iSteps;
    m_fFBOSizeFactor = old_fFBOSizeFactor;
    this->resize(m_iWidth, m_iHeight);
}
