#version 120

uniform sampler3D volume;
uniform sampler1D transfer;
uniform sampler2D front;
uniform sampler2D back;
uniform float dz;
uniform int maxrange;
uniform vec3 texCoordSteps;
uniform mat4 modelView;

vec3 calcPhong(vec4 color, vec3 normal) {	
	
	const vec4 worldViewDir = vec4(0.0,0.0,1.0,0.0);
	const vec4 worldLightDir = vec4(1.0,1.0,1.0,0.0);

	const float ambCoeff = 0.05;
	const vec3 ambLightColor = vec3(1.0, 1.0, 1.0);	
	const vec3 directLightColor = vec3(1.0, 1.0, 1.0);
	const float diffCoeff = 0.6;
	const float specCoeff = 0.45;
	const float specExp = 32;
	
	vec3 viewDirection = normalize(modelView*worldViewDir).xyz;
	vec3 lightDirection = normalize(modelView*worldLightDir).xyz*-1;

	const vec3 ambient = ambCoeff * ambLightColor;

	vec3 diff = diffCoeff*directLightColor*max(0.0, dot(normal, lightDirection));
	
	vec3 reflection = normalize(reflect(lightDirection, normal));
	vec3 spec = specCoeff*directLightColor*min(pow(max(0.0, dot(viewDirection, reflection)), specExp), 1.0);

	return color.xyz*(ambient + diff) + spec;
}

vec3 calcNormal(vec3 point) {

	vec3 xStep = vec3(texCoordSteps.x , 0.0, 0.0);
	vec3 yStep = vec3(0.0, texCoordSteps.y , 0.0); 
	vec3 zStep = vec3(0.0, 0.0, texCoordSteps.z);

	float dx = (texture3D(volume, point+xStep).x - texture3D(volume, point-xStep).x)*0.5;
	float dy = (texture3D(volume, point+yStep).x - texture3D(volume, point-yStep).x)*0.5;
	float dz = (texture3D(volume, point+zStep).x - texture3D(volume, point-zStep).x)*0.5;
	
	vec3 normal = vec3(dx, dy, dz);

	if(length(normal)==0.0)
		return vec3(0.0, 0.0, 0.0);
	else
		return normalize(normal);
}

void main()
{
	// Code nach VisLU-Tutorial (http://www.cg.tuwien.ac.at/~ilcik/VolVisOpenGL2008.pdf)

	vec4 entry_point = texture2D(front, vec2(gl_TexCoord[0]));
	vec4 exit_point = texture2D(back, vec2(gl_TexCoord[0]));
	
	float dist = distance(entry_point, exit_point)/dz;
	int maxiter = int(floor(dist));

	// exit - entry => front-to-back 
	vec3 diff = (exit_point.xyz - entry_point.xyz)/dist;

	if (entry_point.w == 0.0 || maxiter==0)
		discard;
	else
	{
		vec3 point = entry_point.xyz;
		vec4 intensity = texture3D(volume, point);
		vec4 transferred = texture1D(transfer, intensity.x);

		vec3 normal = calcNormal(point);
		transferred.xyz = calcPhong(transferred, normal);

		vec4 Result;
		Result.xyz = transferred.xyz*transferred.w;
		Result.w = transferred.w;

		for(int i = 0; i < maxrange; i++)
		{
			intensity = texture3D(volume, point);
			transferred = texture1D(transfer, intensity.x);
			
			vec3 normal = calcNormal(point);
			transferred.xyz = calcPhong(transferred, normal);

			Result.xyz += (1.0 - Result.w) * transferred.w * transferred.xyz;
			Result.w += (1.0 - Result.w) * transferred.w;
			point += diff;
			if ((Result.w >= 1.0) || (i >= maxiter)) break;
		} 

		gl_FragColor = Result;
	}
}