#include "FlowGeometry.h"
#include "reverseBytes.h"

using namespace std;

FlowGeometry::FlowGeometry()
{
    geometryData = NULL;
}

FlowGeometry::~FlowGeometry()
{
    if (geometryData)
        delete[] geometryData;
}

bool FlowGeometry::readFromFile(char* header, FILE* fp, bool bigEndian)
{
	isFlipped = false;

	//determine the dimensions
	sscanf(header,"SN4DB %d %d %d", &dim[0], &dim[1], &dim[2]);
	std::cout << "Dimensions: " << dim[0] << " x " << dim[1] << " x " << dim[2] << std::endl;
	if (dim[2]!=1) {
		std::cerr << "Invalid Z dimension value." << std::endl;
		return false;
	}

	geometryData = new vec3[dim[0] * dim[1]];
	//read the data and check if everything went fine
	int result = fread(geometryData,sizeof(vec3),dim[0]*dim[1],fp);
	if (result != dim[0] * dim[1]) {
		std::cerr << "+ Error reading grid file." << std::endl << std::endl;
		return false;
	}

	if (bigEndian) {
		for (int j = 0; j < getDimX()*getDimY(); j++) {
			for (int k = 0; k < 3; k++) {
				geometryData[j][k] = reverseBytes<float>(geometryData[j][k]);
			}
		}
	}

	//first vertex
	boundaryMin = vec3(getPos(0));
	//last vertex
	boundaryMax = vec3(getPos((dim[0] * dim[1]) - 1));
	boundarySize = boundaryMax - boundaryMin;

	//if X and Y are swapped... meaning that going in a row the y value increases and x stays the same
	if (getPosY(dim[0] - 1) > (boundaryMin[1] + boundaryMax[1]) * 0.5) {
		//this flag will fix that, anywhere where it's neccerasy... watch it :)
		isFlipped = true;
		std::cout << "Flipped Y and X dimensions." << std::endl;
	} else {
		isFlipped = false;
	}

	std::cout << "X Boundaries: " << boundaryMin[0] << " ... " << boundaryMax[0] << std::endl;
	std::cout << "Y Boundaries: " << boundaryMin[1] << " ... " << boundaryMax[1] << std::endl;

	return true;
}

///returns X index of the last vertex lying left to the position x and the Y index of the last vertex lying under the position y 
int FlowGeometry::getXYvtx(vec3 pos)
{
	int i;
	int j;

	//search for the column left to the vertex
	for (i = 0; (i < dim[0]) && (getPosX(getVtx(i,0)) < pos[0]); i++);

	//search for the row under the vertex
	for (j = 0; (j < dim[1]) && (getPosY(getVtx(0,j)) < pos[1]); j++);

	//return the vertex ID of the found vertex
	return getVtx((i<dim[0]) ? i : dim[0]-1, (j<dim[1]) ? j : dim[1]-1);
}

vec3 FlowGeometry::getPos(int vtxID)
{
	return geometryData[vtxID];
}

float FlowGeometry::getPosX(int vtxID)
{
	return geometryData[vtxID][0];
}

float FlowGeometry::getPosY(int vtxID)
{
	return geometryData[vtxID][1];
}

///a very slow and dumb routine, that finds the nearest vertex to the given position
/// improved by thj using binarySearch
int FlowGeometry::getNearestVtx(vec3 pos)
{
	/*
	int startIndexX = 0;
	int startIndexY = 0;

	int endIndexX = this->getDimX();
	int endIndexY = this->getDimY();

	int refMidIndexX = 0;
	int refMidIndexY = 0;

	while(true)
	{
		refMidIndexX = (endIndexX - startIndexX) / 2 + startIndexX;
		refMidIndexY = (endIndexY - startIndexY) / 2 + startIndexY;

		if(getPosX(getVtx(refMidIndexX,refMidIndexY)) < pos.v[0])
		{
			startIndexX = refMidIndexX;
		}
		else
		{
			endIndexX  = refMidIndexX;
		}

		if(getPosY(getVtx(refMidIndexX, refMidIndexY)) < pos.v[1])
		{
			startIndexY = refMidIndexY;
		}
		else
		{
			endIndexY = refMidIndexY;
		}

		if( abs(endIndexX - startIndexX) <= 1)
			break;
		if( abs(endIndexY - startIndexY) <= 1)
			break;
	}

	float dist1 = (getPos(getVtx(startIndexX, startIndexY))).dist(pos);
	float dist2 = (getPos(getVtx(startIndexX, endIndexY))).dist(pos);
	float dist3 = (getPos(getVtx(endIndexX, endIndexY))).dist(pos);
	float dist4 = (getPos(getVtx(endIndexX, startIndexY))).dist(pos);

	float dist = dist1;
	int nearest = getVtx(startIndexX, startIndexY);

	if(dist2 < dist)
		nearest = getVtx(startIndexX, endIndexY);
	if(dist3 < dist)
		nearest = getVtx(endIndexX, endIndexY);
	if(dist4 < dist)
		nearest = getVtx(endIndexX, startIndexY);
	
	*/

	// initial startIndex at 0,0
	int startIndexX = 0;
	int startIndexY = 0;

	// initial endIndex at dimX, dimY
	int endIndexX = this->getDimX();
	int endIndexY = this->getDimY();

	// midIndices for our reference point
	int refMidIndexX = 0;
	int refMidIndexY = 0;

	// initial distance: take first point
	float refDist = pos.dist(this->getPos(0));

	while (true) {
		// calculate new middle-indices: in half between start and end
		int midIndexX = (startIndexX + endIndexX) / 2;
		int midIndexY = (startIndexY + endIndexY) / 2;

		// get the point in the middle
		vec3 midPos = this->getPos(this->getVtx(midIndexX, midIndexY));
		// calculate the distance between the middle-point and the position we are looking for
		float newDist = pos.dist(midPos);
		// the distance is smaller, mark the actual middle point as our new reference point
		if (newDist < refDist) {
			refDist = newDist;
			refMidIndexX = midIndexX;
			refMidIndexY = midIndexY;
		}

		// x of our point to search is smaller then our middle position => endIndexX is reduced to middle - 1
		if (pos.v[0] < midPos.v[0])
			endIndexX = midIndexX - 1;
		// x of our point to search is bigger then our middle position => startIndexX is increased to middle + 1
		else
			startIndexX = midIndexX + 1;

		// y of our point to search is smaller then our middle position => endIndexY is reduced to middle - 1
		if (pos.v[1] < midPos.v[1])
			endIndexY = midIndexY - 1;
		// y of our point to search is bigger then our middle position => startIndexY is increased to middle + 1
		else
			startIndexY = midIndexY + 1;

		// endIndex smaller or equals startIndex => finished
		if (endIndexX <= startIndexX)
			break;

		// endIndex smaller or equals startIndex => finished
		if (endIndexY <= startIndexY)
			break;
	}



	return (isFlipped) ? (refMidIndexX * dim[1]) + refMidIndexY : (refMidIndexY * dim[0]) + refMidIndexX;

	//return nearest;
}

bool FlowGeometry::getInterpolationAt(vec3 pos, int* vtxID, float* coef)
{
	//if we are outside of the dataset, return false
	//please note, that this test is valid only for rectangular datasets (block, hurricane), but not for tube
	//if ((pos[0] < boundaryMin[0]) || (pos[1] < boundaryMin[1]) || (pos[0] > boundaryMax[0]) || (pos[1] > boundaryMax[1]))
	//	return false;

	//example of a low-quality lookup with no interpolation
	vtxID[0] = getNearestVtx(pos); //finds the nearest vertex to the given position and returns it's value as the dominanting one. The getNearestVtx needs to be improved to gain some speed.
	
	if ( getPosX(vtxID[0]) < pos[0] )
	{
		//vtxID[1] = vtxID[0] + 1;
		//vtxID[1] = getVtx(getPosX(vtxID[0]) + 1, getPosY(vtxID[0]));
		vtxID[1] = getVtx(getVtxX(vtxID[0]) + 1, getVtxY(vtxID[0]));
	}
	else
	{
		//vtxID[1] = vtxID[0] + 1;
		//vtxID[1] = getVtx(getPosX(vtxID[0]) - 1, getPosY(vtxID[0]));
		vtxID[1] = getVtx(getVtxX(vtxID[0]) - 1, getVtxY(vtxID[0]));
	}
	
	if ( getPosY(vtxID[0]) < pos[1] )
	{
		//vtxID[2] = vtxID[0] + getDimX();
		//vtxID[2] = getVtx(getPosX(vtxID[0]), getPosY(vtxID[0]) + 1 );
		vtxID[2] = getVtx(getVtxX(vtxID[0]), getVtxY(vtxID[0]) + 1 );
	}
	else
	{
		//vtxID[2] = vtxID[0] - getDimX();
		//vtxID[2] = getVtx(getPosX(vtxID[0]), getPosY(vtxID[0]) - 1 );
		vtxID[2] = getVtx(getVtxX(vtxID[0]), getVtxY(vtxID[0]) - 1 );
	}
		
	vtxID[3] = getVtx(getVtxX(vtxID[1]), getVtxY(vtxID[2]));

	// Inverse distance weighting (IDW) - Shepard's Method
	float p = 1.0f;		// smoothing factor
	float w[4];		// point weigths
	
	w[0] = 1 / pow( pos.dist( getPos( vtxID[0] ) ) , p );
	w[1] = 1 / pow( pos.dist( getPos( vtxID[1] ) ) , p );
	w[2] = 1 / pow( pos.dist( getPos( vtxID[2] ) ) , p );
	w[3] = 1 / pow( pos.dist( getPos( vtxID[3] ) ) , p );

	coef[0] = w[0] / (w[0] + w[1] + w[2] + w[3]);
	coef[1] = w[1] / (w[0] + w[1] + w[2] + w[3]);
	coef[2] = w[2] / (w[0] + w[1] + w[2] + w[3]);
	coef[3] = w[3] / (w[0] + w[1] + w[2] + w[3]);

// 	vtxID[0] = getNearestVtx(pos);
// 	vtxID[1] = vtxID[0];
// 	vtxID[2] = vtxID[0];
// 	vtxID[3] = vtxID[0];
// 
// 	coef[0] = 1.0f;
// 	coef[1] = 0.0f;
// 	coef[2] = 0.0f;
// 	coef[3] = 0.0f;

	return true;
}

float FlowGeometry::getMinX()
{
	return boundaryMin[0];
}

float FlowGeometry::getMaxX()
{
	return boundaryMax[0];
}

float FlowGeometry::getMinY()
{
	return boundaryMin[1];
}

float FlowGeometry::getMaxY()
{
	return boundaryMax[1];
}

int FlowGeometry::getDimX()
{
	return (isFlipped) ? dim[1] : dim[0];
}

int FlowGeometry::getDimY()
{
	return (isFlipped) ? dim[0] : dim[1];
}

int FlowGeometry::getDimZ()
{
	return dim[2];
}

int FlowGeometry::getVtx(int x, int y)
{
	//if we need to flip the rows and columns, we do it here 
	return (isFlipped)? (x*dim[1]) + y : (y*dim[0]) + x;
}

int FlowGeometry::getVtxX(int vtxID)
{
	//if we need to flip the rows and columns, we do it here
	return (isFlipped)? vtxID / dim[1] : vtxID % dim[0];
}

int FlowGeometry::getVtxY(int vtxID)
{
	//if we need to flip the rows and columns, we do it here
	return (isFlipped)? vtxID % dim[1] : vtxID / dim[0];
}

int FlowGeometry::getRightNeigh(int vtxID)
{
	int x = getVtxX(vtxID);
	return (x+1 < dim[0]) ? getVtx(x+1,getVtxY(vtxID)) : -1;
}

int FlowGeometry::getTopNeigh(int vtxID)
{
	//remember that the data is structured with (0,0) in the upper-left corner and (1,1) in the lower-right
	//that's why we are subtracting 1 to find the top neighbour
	int y = getVtxY(vtxID);
	return (y+1 < dim[1]) ? getVtx(getVtxX(vtxID), y-1) : -1;
}

int FlowGeometry::getLeftNeigh(int vtxID)
{
	int x = getVtxX(vtxID);
	return (x > 1) ? getVtx(x-1,getVtxY(vtxID)) : -1;
}

int FlowGeometry::getBottomNeigh(int vtxID)
{
	//remember that the data is structured with (0,0) in the upper-left corner and (1,1) in the lower-right
	int y = getVtxY(vtxID);
	return (y < 1) ? getVtx(getVtxX(vtxID), y+1) : -1;
}

vec3 FlowGeometry::normalizeCoords(vec3 pos)
{
	vec3 u(pos - boundaryMin);
	//scale each component according to the side length
	u[0] /= boundarySize[0];
	u[1] /= boundarySize[1];

	return u;
}

vec3 FlowGeometry::unNormalizeCoords(vec3 pos)
{
	vec3 u(pos);
	//multiply each component according to the side length
	u[0] *= boundarySize[0];
	u[1] *= boundarySize[1];

	return u += boundaryMin;
}
