#include "FlowData.h"
#include <math.h>
#include "reverseBytes.h"

FlowData::FlowData()
{
	this->loaded = false;
	this->numChannels = 0;
	this->timesteps = 0;
}

FlowData::~FlowData()
{
	for (unsigned int i = 0; i < this->timeChannels.size(); i++)
		delete this->timeChannels[i];
}

bool FlowData::loadDataset(string filename, bool bigEndian)
{
	FILE* griFile = NULL;
	char header[40];

	//localize the last dot in the filename
	unsigned int lastdot = filename.find_last_of(".",filename.length()-1);
	if (lastdot != string::npos) {
		//if there is a dot, remove everything behind it
		filename = filename.substr(0,lastdot);
	}

	/////////////
	// GRID FILE
	/////////////
	string griName = filename + ".gri";
	std::cout << "- Loading grid file '" << griName << "' ... " << std::endl;
	
	//open the grid file
	griFile = fopen(griName.c_str(),"rb");
	if (!griFile) {
		std::cerr << "+ Error loading grid file:" << griName << std::endl << std::endl;
		return false;
	}

	//save the header
	fread(header, 40, 1, griFile);
	//pass the grid file to the geometry class to process it
	if (!geometry.readFromFile(header, griFile, bigEndian))
		return false;

	//close the file
	fclose(griFile);

	int dimX,dimY,dimZ;
	float DT;

	//read some neceassry data from the header
	sscanf(header,"SN4DB %d %d %d %d %d %f",&dimX, &dimY, &dimZ, &this->numChannels, &this->timesteps, &DT);
	this->numChannels += 3; //add the 3 components of the velocity vector to the number of additional chanenls
	printf("Channels: %d\nTimesteps: %d\n", this->numChannels, this->timesteps);

	for (int i = 0; i < this->timesteps; i++) {
		if (this->loadTimestep(filename, i, bigEndian) == false)
			return false;
	}

	return true;
}

bool FlowData::loadTimestep(string filename, int i, bool bigEndian)
{
	/////////////
	// DAT FILE
	/////////////
	//since this framework handles only one timestep, we use only a single dat file

	FILE* datFile = NULL;

	char suffix[16];
	sprintf(suffix,".%.5u.dat", i); //the second dot and the following 5 specify that a minimum of 5 numbers will be written
	string datName = filename.append(suffix);
	std::cout << "- Loading dat file '" << datName << "' ... " << std::endl;

	//open the dat file
	datFile = fopen(datName.c_str(),"rb");
	if (!datFile) {
		std::cout << "+ Error loading dat file: '" << datName << "'" << std::endl << std::endl;
		return false;
	}

	//let's prepare the channels
	int* ch = new int[this->numChannels]; //create a storage for addresses our channels

	//because reading big chunks of data is much faster than single values, 
	//we read the data into a temporary array and then copy it to the channels
	float* tmpArray = new float[this->numChannels * this->geometry.getDimX() * this->geometry.getDimY()]; //create temporary storage
	int result = fread(tmpArray, sizeof(float), this->numChannels * this->geometry.getDimX() * this->geometry.getDimY(), datFile); //read the data
	//have we read the whole data file?
	if (result != this->numChannels * this->geometry.getDimX() * this->geometry.getDimY()) {
		std::cout << "+ Error reading dat file: '" << datName << "'" << std::endl << std::endl;
		return false;
	}

	//close the file, it is no longer needed
	fclose(datFile);

	//if swap the byte order, if the file is encoded big-endian
	if (bigEndian) {
		for(int j = 0; j < this->numChannels * this->geometry.getDimX() * this->geometry.getDimY(); j++)
			tmpArray[j] = reverseBytes<float>(tmpArray[j]);
	}

	TimeChannel* timeChannel = new TimeChannel(&this->geometry);

	//assign the data to the appropriate channels
	for (int j = 0; j < this->numChannels; j++) {
		//create the new channel
		ch[j] = timeChannel->createChannel();
		//copy the values of the jth channel from tmpArray, which carries numChannels    
		timeChannel->channels[ch[j]]->copyValues(tmpArray, this->numChannels, j);
	}

	this->timeChannels.push_back(timeChannel);

	delete[] ch;
	delete[] tmpArray;

	return true;
}
