#ifndef RENDERER_H
#define RENDERER_H

#include <GL/freeglut.h>

#include <string>
#include <vector>


#include "../utils/FlowData.h"
#include "../utils/StreamlineGrid.h"
#include "../utils/vec3.h"

#include <semaphore.h>

/**
* @class Renderer encapsulates the rendering pipeline both for the Slicing-Rendering and Volume-Rendering of the Volumes.
*/
class Renderer
{
 public:
	/**
	* @class TFPoint holds the information for one position on the Transferfunction
	*/
	struct TFPoint
	{
		unsigned int pos; //! @param pos defines the position of this RGBA quadrupple inside the Transferfunction. Ranges between 0 and the width of the Transferfunction
		unsigned char r; //! @param r defines the r (Red) value of this position inside the Transferfunction. Ranges from 0 to 255
		unsigned char g; //! @param g defines the g (Green) value of this position inside the Transferfunction. Ranges from 0 to 255
		unsigned char b; //! @param b defines the b (Blue) value of this position inside the Transferfunction. Ranges from 0 to 255
		unsigned char a; //! @param a defines the a (Alpha) value of this position inside the Transferfunction. Ranges from 0 to 255
	};

	/**
	* @class TransferFunction holds all Data which is needed to represent a Transferfunction
	*/
	struct TransferFunction
	{
		GLuint texID; //! @param texID stores the id of the texture this Transferfunction is bound when in use by OpenGL
		unsigned int width; //! @param width defines the x-size in pixel this Transferfunction has as a 1D texture
		std::vector<TFPoint> points; //! @param points holds the Points which spawn the Transferfunction and between whose will be linearly interpolated to generate the Texturedata
		int* histo;
		int* maxHist;

		unsigned char* data; //! @param data holds a pointer to the already generated and interpolated Texturedata to be used by OpenGL

		TransferFunction()
		{
			this->width = 0;
			this->texID = 0;
			this->data = 0;
			//this->points = new std::vector<TFPoint>();
		}
	};

	//! @brief one must not directly instantiate the Renderer, VolumeRendering will provide an instance
	Renderer();
	Renderer(int);
	~Renderer();

	void setStreamlines(bool);
	void setArrows(bool);

	void setArrowSize(float);

	void setDTest(float);
	void setDSep(int);
	void setStepSize(float);

	void setDataset(int);
	void setChannel(int);

	//! @brief sets a Transferfunction to be used by the rendering
	void setTransferFunction(TransferFunction&);

	//! @brief is being called to initalize the Renderer. MUST NOT BE CALLED, is done by RenderWindow
	void initGL();
	//! @brief is being called to to resize the Viewport. MUST NOT BE CALLED, is done by RenderWindow
	void resizeGL(int, int);
	//! @brief is being called regulary to keep up the Rendering-Loop. MUST NOT BE CALLED, is done by RenderWindow
	void paintGL();

	//! @brief is being called to process animation-steps when animation is active. MUST NOT BE CALLED, is done by RenderWindow
	void animate();

	int * getHisto() { return this->histo; };
	int * getMaxHist() { return this->maxHist; };

 private:
	static Renderer* instance;

	//! @brief the factor how long the last rendering step took
	double loopFactor;
	clock_t timestamp;

	FlowData* dataset;

	int datasetIndex;

	int* histo;

	int* maxHist;

	//! @brief width of the viewport
	GLuint width;
	//! @brief height of the viewport
	GLuint height;

	int channelIndex;

	int timeStepIndex;

	int xCountArrows;
	int yCountArrows;

	float xArrowDist;
	float yArrowDist;

	float arrowSize;

	//! @brief OpenGL-ID of the Transferfunction-Texture
	GLuint transferTexture;

	//! @brief OpenGL-ID of the uniform location of the Transferfunction-Texture in the Volume-Rendering Shader
	GLint transferTextureLocation;

	GLuint arrowTexture;
	GLint arrowTextureLocation;

	GLint arrowPlottingProgram;
	GLint arrowPlottingChannelTexturesLocations[2];

	GLint colorCodingProgram;
	GLint colorCodingChannelTextureLocation;

	vector<GLuint*> timeStepTextures;
	vector<vector<float*> > timeStepData;

	float newStepSize;
	int newDSep;

	float stepSize;
	int dSep;
	StreamlineGrid* streamlineGrid;

	bool doArrows;
	bool doStreamlines;

	void loadArrowTexture();
	void preprocessFlowData();

	void renderChannel();
	void renderArrows();
	void renderStreamlines();

	//! @brief checks if all OpenGL features are present on this hardware neede to to Volume-Rendering as implemented hiere
	void checkGLFeatures();

	static bool create2DTexture(GLuint*, int, int, void*);

	//! @brief creates a Shader-Program out of two given files: the vertexprogram and the fragmentprogram
	static GLuint createProgram(const std::string&, const std::string&);
	//! @brief queries and returns uniform locations
	static GLint queryUniformLoc(GLint, const GLchar*);
	//! @brief reads text from a file to load shader sourcefiles
	static bool readShaderSource(const std::string&, std::string&);

	sem_t mutex;

	static void* generateTimestepTextures(void*);
};

#endif
