#include <sys/time.h>

#include <GL/glew.h>
#include <GL/freeglut.h>

#include "Renderer.h"

#include <QImage>
#include <QGLWidget>

#include <vector>
#include <iostream>
#include <string>

#include <assert.h>

#include <string.h>
#include <math.h>
#include <stdlib.h>

#include <pthread.h>
#include <semaphore.h>

#define MAX_PREPROCESSING_THREADS 4

using namespace std;

Renderer* Renderer::instance = 0;

//! @brief prints the shader information log for a shader object
void printShaderInfoLog(GLuint obj)
{
	int infologLength = 0;
	int charsWritten = 0;
	char* infoLog;

	glGetShaderiv(obj, GL_INFO_LOG_LENGTH, &infologLength);

	if (infologLength > 0) {
		infoLog = (char*) malloc(infologLength);
		glGetShaderInfoLog(obj, infologLength, &charsWritten, infoLog);
		if (charsWritten)
			printf("%s\n",infoLog);
		free(infoLog);
	}
}

//! @brief prints the info log of a program object
void printProgramInfoLog(GLuint obj)
{
	int infologLength = 0;
	int charsWritten  = 0;
	char *infoLog;

	glGetProgramiv(obj, GL_INFO_LOG_LENGTH,&infologLength);
	if (infologLength > 0) {
	        infoLog = (char*) malloc(infologLength);
	        glGetProgramInfoLog(obj, infologLength, &charsWritten, infoLog);
		if (charsWritten)
			printf("%s\n",infoLog);
	        free(infoLog);
	}
}

int GetNextPowerOfTwo(const int iNumber)
{
	int iPowerOfTwo = 1;

	while (iPowerOfTwo < iNumber)
		iPowerOfTwo = iPowerOfTwo * 2;

	return iPowerOfTwo;
};

Renderer::Renderer()
{
	this->datasetIndex = 4;
	this->arrowSize = 10.0;
	this->xArrowDist = 1.5;
	this->yArrowDist = 1.5;

	this->dSep = 10;
	this->stepSize = 5;

	this->newStepSize = this->stepSize;
	this->newDSep = this->dSep;

	this->doArrows = false;
	this->doStreamlines = false;

	this->channelIndex = 0;

	this->timeStepIndex = 0;
	this->timestamp = 0;

	this->xCountArrows = 20;
	this->yCountArrows = 20;

	this->dataset = 0;

	this->histo = 0;
	this->maxHist = new int(0);

	this->width = 0;
	this->height = 0;

	this->transferTexture = 0;

	this->transferTextureLocation = 0;

	sem_init(&this->mutex, 0, 1);
	Renderer::instance = this;
}

Renderer::Renderer(int index)
{
	this->datasetIndex = index;
	this->arrowSize = 10.0;
	this->xArrowDist = 1.5;
	this->yArrowDist = 1.5;

	this->dSep = 10;
	this->stepSize = 5;

	this->newStepSize = this->stepSize;
	this->newDSep = this->dSep;

	this->doArrows = false;
	this->doStreamlines = false;

	this->channelIndex = 0;

	this->timeStepIndex = 0;
	this->timestamp = 0;

	this->xCountArrows = 20;
	this->yCountArrows = 20;

	this->dataset = 0;

	this->histo = 0;
	this->maxHist = new int(0);

	this->width = 0;
	this->height = 0;

	this->transferTexture = 0;

	this->transferTextureLocation = 0;

	sem_init(&this->mutex, 0, 1);
	Renderer::instance = this;
}

Renderer::~Renderer()
{
	Renderer::instance = 0;

	sem_destroy(&this->mutex);
}

void Renderer::setArrowSize(float arrowSize)
{
	this->arrowSize = arrowSize;
}

void Renderer::setDSep(int dSep)
{
	this->newDSep = dSep;
}

void Renderer::setDTest(float dTest)
{
	if (this->streamlineGrid == 0)
		return;

	this->streamlineGrid->changeDTest(dTest);
}

void Renderer::setStepSize(float stepSize)
{
	this->newStepSize = stepSize;
}

void Renderer::setStreamlines(bool flag)
{
	this->doStreamlines = flag;
}

void Renderer::setArrows(bool flag)
{
	this->doArrows = flag;
}

void Renderer::setChannel(int chIndex)
{
	if (chIndex < this->dataset->getChannelsNum())
	{
		this->channelIndex = chIndex;
	}
}

void Renderer::setDataset(int datasetIndex)
{
	this->dataset = new FlowData();
	bool flag = false;

	this->arrowSize = 10.0;
	this->xArrowDist = 1.5;
	this->yArrowDist = 1.5;

	this->dSep = 10;
	this->stepSize = 5;

	this->newStepSize = this->stepSize;
	this->newDSep = this->dSep;

	this->doArrows = false;
	this->doStreamlines = false;

	this->channelIndex = 0;

	this->timeStepIndex = 0;
	this->timestamp = 0;

	this->xCountArrows = 20;
	this->yCountArrows = 20;

	this->histo = 0;
	this->maxHist = new int(0);

	switch(datasetIndex)
	{
		case 0:
			flag = this->dataset->loadDataset("data/hurricane_timeseries/hurricane_p_tc", false);
			break;
		case 1:
			flag = this->dataset->loadDataset("data/hurricane10/hurricane_p_tc_singletime10", false);
			break;
		case 2:
			flag = this->dataset->loadDataset("data/hurricane48/hurricane_p_tc_singletime48", false);
			break;
		case 3:
			flag = this->dataset->loadDataset("data/tube/tube", false);
			break;
		case 4:
			flag = this->dataset->loadDataset("data/blockData/c_block", false);
			break;
		default:
			flag = this->dataset->loadDataset("data/hurricane_timeseries/hurricane_p_tc", false);
			break;
	}
	if (flag == false) {
		cout << "ERROR loading DataSet - Program will exit" << endl;
	}

	cout << "Loading FlowData finished" << endl;

	this->loadArrowTexture();

	this->preprocessFlowData();

	this->renderChannel();
}

void Renderer::animate()
{
	// switch through the timesteps when there are more then 1
	if (this->dataset != 0)
	{
		if (this->dataset->getNumTimesteps() > 1) {
			timeval t;
			gettimeofday(&t, 0);
	
			// overflow
			if (this->timestamp > t.tv_usec)
				this->timestamp = t.tv_usec;
		
			// interval: 50ms
			if (t.tv_usec - this->timestamp > 70000) {
				if (this->timeStepIndex == this->dataset->getNumTimesteps() - 1) {
					this->timeStepIndex = 0;
					
					
				} else {
					this->timeStepIndex++;
				}
	
				this->timestamp = t.tv_usec;
			}
		}
	}
}

void Renderer::setTransferFunction(TransferFunction& tf)
{
	cout << "setting new TransferFunction..." << endl;

	// allocate memory for the transferfunction texture
	tf.data = (unsigned char*) malloc(4 * tf.width * sizeof(unsigned char));
	memset(tf.data, 0, 4 * tf.width * sizeof(unsigned char));
	
	tf.histo = this->histo;
	tf.maxHist = this->maxHist;
	
	// first point is initial point
	TFPoint prevPoint = tf.points[0];

	std::cout << "( " << (int) prevPoint.r << " , " << (int) prevPoint.g << " , " << (int) prevPoint.b << " , " << (int) prevPoint.a << " ) @ " << (int) prevPoint.pos << std::endl;

	// iterate over all points an linearly interpolate between prev and this
	for (unsigned int i = 1; i < tf.points.size(); i++) {
		TFPoint thisPoint = tf.points[i];

		std::cout << "( " << (int) thisPoint.r << " , " << (int) thisPoint.g << " , " << (int) thisPoint.b << " , " << (int) thisPoint.a << " ) @ " << (int) thisPoint.pos << std::endl;
		unsigned int distance = thisPoint.pos - prevPoint.pos;

		float factor = 0;
		unsigned int index = 0;
		for (unsigned int j = 0; j < distance; j++) {
			index = (prevPoint.pos + j) * 4;
			factor = (float) (j + 1) / (float) distance;

			tf.data[index + 0] = prevPoint.r + (thisPoint.r - prevPoint.r) * factor;
			tf.data[index + 1] = prevPoint.g + (thisPoint.g - prevPoint.g) * factor;
			tf.data[index + 2] = prevPoint.b + (thisPoint.b - prevPoint.b) * factor;
			tf.data[index + 3] = prevPoint.a + (thisPoint.a - prevPoint.a) * factor;
		}

		prevPoint = thisPoint;
	}

	// there has been already a transfertexture set, delete it from GPU
	if (this->transferTexture) {
		cout << "Unloading old Transferfunction..." << endl;
		glDeleteTextures(1, &this->transferTexture);
		cout << "Unloading old Transferfunction succeeded" << endl;
	}

	GLenum glError = GL_NO_ERROR;
	glError = glGetError();
	
	// generate new texid
	glGenTextures(1, &this->transferTexture);

	glError = glGetError();
	if (glError != GL_NO_ERROR)
		cout << "glGenTextures for 1D failed with " << gluErrorString(glError) << endl;

	// bind texid to gpu
	glBindTexture(GL_TEXTURE_1D, this->transferTexture);

	glError = glGetError();
	if (glError != GL_NO_ERROR)
		cout << "glBindTexture for 1D failed with " << gluErrorString(glError) << endl;
	
	// set filter params
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glError = glGetError();
	if (glError != GL_NO_ERROR)
		cout << "glTexParameteri for 1D failed with " << gluErrorString(glError) << endl;

	// upload the data for this transfertexture by creating mipmaps for it
	gluBuild1DMipmaps(GL_TEXTURE_1D, GL_RGBA16, tf.width, GL_RGBA, GL_UNSIGNED_BYTE, tf.data);

	tf.texID = this->transferTexture;

	cout << "setting new TransferFunction succeeded" << endl;
}

void Renderer::initGL()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glShadeModel(GL_SMOOTH);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClearDepth(1.0f);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glEnable(GL_CULL_FACE);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glHint(GL_LINE_SMOOTH_HINT, GL_DONT_CARE);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glViewport(0, 0, this->width, this->height);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Initialize GLEW
	std::cout << "- Initializing GLEW ..." << std::endl;
	GLenum err = glewInit();
	if (err != GLEW_OK) {
		// glewInit failed, something is seriously wrong
		std::cerr << "Error initializing GLEW: " << glewGetErrorString(err) << std::endl;
		exit(1);
	}

	this->checkGLFeatures();

	cout << "Creating ColorCoding-Program..." << endl;
	this->colorCodingProgram = Renderer::createProgram("shaders/colorCodingVert.glsl", "shaders/colorCodingFrag.glsl");
	if (this->colorCodingProgram) {
		cout << "Creating ColorCoding-Program OK" << endl;

		this->colorCodingChannelTextureLocation = Renderer::queryUniformLoc(this->colorCodingProgram, "channelTex");
		this->transferTextureLocation = Renderer::queryUniformLoc(this->colorCodingProgram, "texTransfer");

	} else {
		cout << "Creating ColorCoding-Program Failed" << endl;
		exit(1);
	}

	cout << "Creating ArrowPlotting-Program..." << endl;
	this->arrowPlottingProgram = Renderer::createProgram("shaders/arrowPlotVert.glsl", "shaders/arrowPlotFrag.glsl");
	if (this->arrowPlottingProgram) {
		cout << "Creating ArrowPlotting-Program OK" << endl;

		this->arrowTextureLocation = Renderer::queryUniformLoc(this->arrowPlottingProgram, "arrowTex");

		// just use the velocity-x and velocity-y channels
		for (int i = 0; i < 2; i++) {
			char suffix[16];
			sprintf(suffix,"channel%.1uTex", i);
			string uniformName(suffix);
			
			this->arrowPlottingChannelTexturesLocations[i] = Renderer::queryUniformLoc(this->arrowPlottingProgram, uniformName.c_str());
		}

	} else {
		cout << "Creating ArrowPlotting-Program Failed" << endl;
		exit(1);
	}

	cout << "Loading FlowData..." << endl;

	this->dataset = new FlowData();
	
	bool flag = false;

	switch(this->datasetIndex)
	{
		case 0:
			flag = this->dataset->loadDataset("data/hurricane_timeseries/hurricane_p_tc", false);
			break;
		case 1:
			flag = this->dataset->loadDataset("data/hurricane10/hurricane_p_tc_singletime10", false);
			break;
		case 2:
			flag = this->dataset->loadDataset("data/hurricane48/hurricane_p_tc_singletime48", false);
			break;
		case 3:
			flag = this->dataset->loadDataset("data/tube/tube", false);
			break;
		case 4:
			flag = this->dataset->loadDataset("data/blockData/c_block", false);
			break;
		default:
			flag = this->dataset->loadDataset("data/hurricane_timeseries/hurricane_p_tc", false);
			break;
	}
	//bool flag = this->dataset->loadDataset("data/hurricane_timeseries/hurricane_p_tc", false);
	//bool flag = this->dataset->loadDataset("data/hurricane10/hurricane_p_tc_singletime10", false);
	//bool flag = this->dataset->loadDataset("data/hurricane48/hurricane_p_tc_singletime48", false);
	//bool flag = this->dataset->loadDataset("data/tube/tube", false);
	//bool flag = this->dataset->loadDataset("data/blockData/c_block", false);
	if (flag == false) {
		cout << "ERROR loading initial DataSet - Program will exit" << endl;
		exit(1);
	}

	cout << "Loading FlowData finished" << endl;

	this->loadArrowTexture();

	this->preprocessFlowData();
}

void Renderer::loadArrowTexture()
{
	QImage arrowImage, arrowGLImage;
	arrowImage.load("data/arrow.png");
	arrowGLImage = QGLWidget::convertToGLFormat(arrowImage);

	GLenum glError = GL_NO_ERROR;
	glError = glGetError();
	
	// generate new texid
	glGenTextures(1, &this->arrowTexture);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glGenTextures for 2D failed with " << gluErrorString(glError) << endl;
		return;
	}

	// bind texid to gpu
	glBindTexture(GL_TEXTURE_2D, this->arrowTexture);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glBindTexture for 2D failed with " << gluErrorString(glError) << endl;
		return;
	}

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	// need to clamp to edge, otherwise it could lead to artifacts at the back of the volume: repeating of intensities
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glTexParameteri for 2D failed with " << gluErrorString(glError) << endl;
		return;
	}

	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA16, arrowGLImage.width(), arrowGLImage.height(), GL_RGBA, GL_UNSIGNED_BYTE, arrowGLImage.bits());
}

void Renderer::preprocessFlowData()
{
	cout << endl;

	cout << "Preprocessing FlowData..." << endl;

	int dimX = this->dataset->getDimX();
	int dimY = this->dataset->getDimY();

	/*
	this->arrowSize = (float) dimX / (float) this->xCountArrows;
	if (dimY / this->yCountArrows < this->arrowSize)
		this->arrowSize = (float) dimY / (float) this->yCountArrows;
	this->arrowSize /= 1.5;
	*/
	this->timeStepData.clear();
	for (int i = 0; i < this->dataset->getNumTimesteps(); i++)
		this->timeStepData.push_back(vector<float*>());

	vector<pthread_t> handles;
	for (int i = 0; i < MAX_PREPROCESSING_THREADS; i++) {
		pthread_t handle;

		pthread_attr_t attr;
		pthread_attr_init(&attr);

		if (pthread_create(&handle, &attr, Renderer::generateTimestepTextures, new int(i)) != 0) {
			cout << "ERROR creating Input-Thread" << endl;
			exit(1);
		}

		handles.push_back(handle);
	}
	std::cout << "1" << std::endl;
	sleep(2);
	std::cout << "2" << std::endl;
	vector<pthread_t>::iterator iter = handles.begin();
	std::cout << "3" << std::endl;
	while (iter != handles.end()) {
		void* returnArg;
		pthread_join(*iter, &returnArg);

		usleep(500000);

		handles.erase(iter);
		iter = handles.begin();
	}
	std::cout << "4" << std::endl;
	this->timeStepTextures.clear();
	for (unsigned int i = 0; i < timeStepData.size(); i++) {
		vector<float*>& timestep = timeStepData[i];
		GLuint* channelTextures = (GLuint*) malloc(timestep.size() * sizeof(GLuint));
		
		for (unsigned int j = 0; j < timestep.size(); j++) {
			Renderer::create2DTexture(&channelTextures[j], dimX, dimY, timestep[j]);
			//TODO: check if needed / find fix - seg fault if used
			delete timestep[j];
		}

		this->timeStepTextures.push_back(channelTextures);
	}
	std::cout << "5" << std::endl;

	this->streamlineGrid = new StreamlineGrid(dimX, dimY, this->dSep);

	std::cout << "6" << std::endl;
	cout << "Preprocessing FlowData finished" << endl;
}

void Renderer::paintGL()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

	timeval t;
	gettimeofday(&t, 0);
	long long startTicks = (t.tv_sec * 1000) + t.tv_usec;

	this->renderChannel();

	if (this->doStreamlines)
		this->renderStreamlines();

	if (this->doArrows)
		this->renderArrows();

	gettimeofday(&t, 0);
	long long endTicks = (t.tv_sec * 1000) + t.tv_usec;

	this->loopFactor = (double)(endTicks - startTicks) / (double) 1000;
}

void Renderer::renderChannel()
{
	glEnable(GL_TEXTURE_2D);

	glUseProgram(this->colorCodingProgram);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, this->timeStepTextures[this->timeStepIndex][this->channelIndex]);
	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_1D, this->transferTexture);

	glUniform1i(this->colorCodingChannelTextureLocation, 0);
	glUniform1i(this->transferTextureLocation, 1);

	int dimX = this->dataset->getDimX();
	int dimY = this->dataset->getDimY();

	//FlowGeometry& geometry = this->dataset->getGeometry();

	// draw a quad with the size of the viewport
	glBegin(GL_QUADS);
// 		for(int x; x < dimX; x++)
// 		{
// 			for(int y; y <= dimY; y++)
// 			{
// 				glTexCoord2f(x / dimX, y / dimY);
// 				glVertex2f(geometry.getPosX(geometry.getVtx(x,y)) / geometry.getMaxX() * dimX, geometry.getPosY(geometry.getVtx(x,y)) / geometry.getMaxY() * dimY);
// 				glTexCoord2f((x + 1) / dimX, y / dimY);
// 				glVertex2f(geometry.getPosX(geometry.getVtx(x + 1,y)) / geometry.getMaxX() * dimX, geometry.getPosY(geometry.getVtx(x+ 1,y)) / geometry.getMaxY() * dimY);
// 				
// 			}
// 		}
		glTexCoord2f(0.0, 0.0); glVertex2f(0, 0);
		glTexCoord2f(0.0, 1.0); glVertex2f(0, dimY);
		glTexCoord2f(1.0, 1.0); glVertex2f(dimX, dimY);
		glTexCoord2f(1.0, 0.0); glVertex2f(dimX, 0);
	glEnd();

	glUseProgram(0);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_1D, 0);

	glDisable(GL_TEXTURE_2D);
}

void Renderer::renderArrows()
{
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_DONT_CARE);

	glUseProgram(this->arrowPlottingProgram);

	int dimX = this->dataset->getDimX();
	int dimY = this->dataset->getDimY();

	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, this->timeStepTextures[this->timeStepIndex][0]);
	glActiveTexture(GL_TEXTURE2);
	glBindTexture(GL_TEXTURE_2D, this->timeStepTextures[this->timeStepIndex][1]);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, this->arrowTexture);

	glUniform1i(this->arrowTextureLocation, 0);
	glUniform1i(this->arrowPlottingChannelTexturesLocations[0], 1);
	glUniform1i(this->arrowPlottingChannelTexturesLocations[1], 2);
	GLint sizeLoc;
	
	sizeLoc = glGetUniformLocation(this->arrowPlottingProgram, "pointSize");
	glUniform1f(sizeLoc, this->arrowSize);

	glEnable(GL_POINT_SPRITE);
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
	glTexEnvi(GL_POINT_SPRITE, GL_COORD_REPLACE, GL_TRUE);
	glPointSize(this->arrowSize);

	this->xCountArrows = (float) dimX / ( (float) this->arrowSize * xArrowDist);
	this->yCountArrows = (float) dimY / ( (float) this->arrowSize * yArrowDist);

	glBegin(GL_POINTS);

	for (int y = 0; y < this->yCountArrows; y++) {
		for (int x = 0; x < this->xCountArrows; x++) {
			float xCoord = this->arrowSize + (x * (dimX / this->xCountArrows));
			float yCoord = this->arrowSize + (y * (dimY / this->yCountArrows));
			glVertex3f(xCoord, yCoord, 1);
		}
	}

	glEnd();

	glDisable(GL_POINT_SPRITE);
	glDisable(GL_VERTEX_PROGRAM_POINT_SIZE);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE2);
	glBindTexture(GL_TEXTURE_2D, 0);

	glUseProgram(0);

	glDisable(GL_TEXTURE_2D);

}

void Renderer::renderStreamlines()
{
	glDisable(GL_TEXTURE_2D);

	int dimX = this->dataset->getDimX();
	int dimY = this->dataset->getDimY();

	if (this->dSep != this->newDSep) {
		this->dSep = this->newDSep;
		this->streamlineGrid->changeDSep(this->dSep);
	}

	if (this->newStepSize != this->stepSize)
		this->stepSize = this->newStepSize;

	FlowGeometry& geometry = this->dataset->getGeometry();
	FlowChannel* velocityXChannel = this->dataset->getTimeChannel(this->timeStepIndex)->getChannel(0);
	FlowChannel* velocityYChannel = this->dataset->getTimeChannel(this->timeStepIndex)->getChannel(1);

	float geomMinX = geometry.getMinX();
	float geomMinY = geometry.getMinY();
	float geomMaxX = geometry.getMaxX();
	float geomMaxY = geometry.getMaxY();

	this->streamlineGrid->clear();

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_DONT_CARE);
	glLineWidth(1.0);

	while (true) {
		
		StreamLine* streamLine = this->streamlineGrid->getNextLine();
		if (streamLine == 0)
			break;

		vec3 frontPos(streamLine->getStartSample());
		vec3 backPos(streamLine->getStartSample());
	
		bool doFront = true;
		bool doBack = true;

		while (true) {
			// still doing front-integration...
			if (doFront) {
				// transfor our pixel-position to the geometry-position
				float x = (frontPos[0] / dimX) * geometry.getMaxX(); // * (geomMaxX - geomMinX) + geomMinX;
				float y = (frontPos[1] / dimY) * geometry.getMaxY(); // * (geomMaxY - geomMinY) + geomMinY;

				vec3 geomFrontPos(x, y);
				vec3 velocityFront(velocityXChannel->getValue(geomFrontPos), velocityYChannel->getValue(geomFrontPos));

				// need to normalize
				velocityFront.normalize();
	
				// apply stepsize
				velocityFront *= stepSize;
	
				// calculate new position
				vec3 newFrontPos(frontPos);
				newFrontPos += velocityFront;
	
				// check boundaries
				if (newFrontPos[0] < 0 || newFrontPos[0] > dimX)
					doFront = false;
				if (newFrontPos[1] < 0 || newFrontPos[1] > dimY)
					doFront = false;
				
				// check if the point is valid
				if (streamLine->addSample(newFrontPos) == false)
					doFront = false;

				if (doFront) {
					// draw lines and update active position
					glBegin(GL_LINES);
						glColor3f(1.0, 1.0, 1.0); glVertex2f(frontPos[1], frontPos[0]);
						glColor3f(1.0, 1.0, 1.0); glVertex2f(newFrontPos[1], newFrontPos[0]);
					glEnd();
		
					frontPos = newFrontPos;
				}
			}
	
			// still doing back-integration...
			if (doBack) {
				// transfor our pixel-position to the geometry-position
				float x = (backPos[0] / dimX) * geometry.getMaxX(); // * (geomMaxX - geomMinX) + geomMinX;
				float y = (backPos[1] / dimY) * geometry.getMaxY(); // * (geomMaxX - geomMinX) + geomMinX;
	
				vec3 geomBackPos(x, y);
		
				// negate the values because inverted integration
				vec3 velocityBack(-velocityXChannel->getValue(geomBackPos), -velocityYChannel->getValue(geomBackPos));
		
				// need to normalize
				velocityBack.normalize();
		
				// apply stepsize
				velocityBack *= stepSize;
		
				// calculate new position
				vec3 newBackPos(backPos);
				newBackPos += velocityBack;

				// check boundaries
				if (newBackPos[0] < 0 || newBackPos[0] > dimX)
					doBack = false;
				if (newBackPos[1] < 0 || newBackPos[1] > dimY)
					doBack = false;

				// check if the point is valid
				if (streamLine->addSample(newBackPos) == false)
					doBack = false;

				if (doBack) {
					// draw lines and update active position
					glBegin(GL_LINES);
						glColor3f(1.0, 1.0, 1.0); glVertex2f(backPos[1], backPos[0]);
						glColor3f(1.0, 1.0, 1.0); glVertex2f(newBackPos[1], newBackPos[0]);
					glEnd();
		
					backPos = newBackPos;
				}
			}
	
			// no more to do: exit loop
			if (!doFront && !doBack)
				break;
		}

		this->streamlineGrid->addStreamLine(streamLine);
	}

	glDisable(GL_LINE_SMOOTH);
}

void Renderer::resizeGL(int w, int h)
{
	this->width = w;
	this->height = h;

	glViewport(0, 0, this->width, this->height);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glOrtho(0.0f, this->width, this->height, 0.0f, -1.0f, 1.0f);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

bool Renderer::create2DTexture(GLuint* id, int x, int y, void* data)
{
	GLenum glError = GL_NO_ERROR;
	glError = glGetError();
	
	if(glIsTexture(*id))
		glDeleteTextures(1,id);

	// generate new texid
	glGenTextures(1, id);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glGenTextures for 2D failed with " << gluErrorString(glError) << endl;
		return false;
	}

	// bind texid to gpu
	glBindTexture(GL_TEXTURE_2D, *id);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glBindTexture for 2D failed with " << gluErrorString(glError) << endl;
		return false;
	}

/*
	glTexImage2D(GL_TEXTURE_2D, 0, 1, x, y, 0, GL_LUMINANCE, GL_FLOAT, data);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glTexImage2D for 2D failed with " << gluErrorString(glError) << endl;
		return false;
	}

	// provide data of the texture
	glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, arrowImage.width(), arrowImage.height(), GL_RGBA, GL_UNSIGNED_BYTE, (void*) arrowImage.bits());
	glError = glGetError();
	if (glError != GL_NO_ERROR)
		cout << "glTexSubImage2D for 2D failed with " << gluErrorString(glError) << endl;
*/

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	// need to clamp to edge, otherwise it could lead to artifacts at the back of the volume: repeating of intensities
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);

	glError = glGetError();
	if (glError != GL_NO_ERROR) {
		cout << "glTexParameteri for 2D failed with " << gluErrorString(glError) << endl;
		return false;
	}

	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA16, x, y, GL_LUMINANCE, GL_FLOAT, data);

	return true;
}

GLuint Renderer::createProgram(const string& vertexSourceFile, const string& fragmentSourceFile)
{
	GLint status;
	GLuint program = 0;
	GLuint vertexShader = 0;
	GLuint fragmentShader = 0;

	string vertexSourceStr;
	string fragmentSourceStr;

	if (Renderer::readShaderSource(vertexSourceFile, vertexSourceStr) == false)
		return 0;
	
	if (Renderer::readShaderSource(fragmentSourceFile, fragmentSourceStr) == false)
		return 0;
	
	const char* vertexSource = vertexSourceStr.c_str();
	const char* fragmentSource = fragmentSourceStr.c_str();

	vertexShader = glCreateShader(GL_VERTEX_SHADER);
	if (vertexShader == 0) {
		cout << "glCreateShader for GL_VERTEX_SHADER \"" << vertexSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
		return 0;
	}
	fragmentShader = glCreateShader(GL_FRAGMENT_SHADER);
	if (fragmentShader == 0) {
		cout << "glCreateShader for GL_FRAGMENT_SHADER \"" << fragmentSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
		return 0;
	}

	glShaderSource(vertexShader, 1, (const GLchar**) &vertexSource, NULL);
	if (glGetError() != GL_NO_ERROR) {
		cout << "glShaderSource for GL_VERTEX_SHADER \"" << vertexSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
		return 0;
	}
	glShaderSource(fragmentShader, 1, (const GLchar**) &fragmentSource, NULL);
	if (glGetError() != GL_NO_ERROR) {
		cout << "glShaderSource for GL_FRAGMENT_SHADER \"" << fragmentSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
		return 0;
	}

	glCompileShader(vertexShader);
	if (glGetError() != GL_NO_ERROR)
		cout << "glCompileShader for GL_VERTEX_SHADER \"" << vertexSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
	glGetShaderiv(vertexShader, GL_COMPILE_STATUS, &status);
	printShaderInfoLog(vertexShader);
	if (!status) {
		cout << "Failed compiling GL_VERTEX_SHADER \"" << vertexSourceFile << "\"" << endl;
		return 0;
	}

	glCompileShader(fragmentShader);
	if (glGetError() != GL_NO_ERROR)
		cout << "glCompileShader for GL_FRAGMENT_SHADER \"" << fragmentSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
	glGetShaderiv(fragmentShader, GL_COMPILE_STATUS, &status);
	printShaderInfoLog(fragmentShader);
	if (!status) {
		cout << "Failed compiling GL_FRAGMENT_SHADER \"" << fragmentSourceFile << "\"" << endl;
		return 0;
	}

	program = glCreateProgram();
	if (program == 0) {
		if (glGetError() != GL_NO_ERROR)
			cout << "glCreateProgram failed with " << gluErrorString(glGetError()) << endl;

		return 0;
	}

	glAttachShader(program, vertexShader);
	if (glGetError() != GL_NO_ERROR)
		cout << "glCompileShader for GL_VERTEX_SHADER \"" << vertexSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;
	glAttachShader(program, fragmentShader);
	if (glGetError() != GL_NO_ERROR)
		cout << "glCompileShader for GL_FRAGMENT_SHADER \"" << fragmentSourceFile << "\" failed with " << gluErrorString(glGetError()) << endl;

	glLinkProgram(program);
	if (glGetError() != GL_NO_ERROR)
		cout << "glLinkProgramARB failed with " << gluErrorString(glGetError()) << endl;
	glGetProgramiv(program, GL_LINK_STATUS, &status);
	printProgramInfoLog(program);
	if (!status) {
		cout << "ERROR linking of program \"" << vertexSourceFile << "\" and \"" << fragmentSourceFile << "\" failed" << endl;
		return 0;
	}

	return program;
}

bool Renderer::readShaderSource(const string& file, string& shaderSource)
{
	string fullFileName = file;
	
	FILE* shaderSourceFile = fopen(fullFileName.c_str(), "r");
	if (shaderSourceFile == 0) {
		cout << "ERROR ... couldn't open Shadersource-File " << fullFileName << endl;
		return false;
	}
	
	char c;
	while ((c = fgetc(shaderSourceFile)) != EOF)
		shaderSource += c;
	
	return true;
}

GLint Renderer::queryUniformLoc(GLint prog, const GLchar* name)
{
	GLint location = 0;
	
	location = glGetUniformLocation(prog, name);
	if (location == -1) {
		cout << "Coulnd't get Uniform Location for name \"" << name << "\". OpenGL-Error: " << gluErrorString(glGetError())  << endl;
		return -1;
	}

	return location;
}

void Renderer::checkGLFeatures()
{
	if (!GLEW_VERSION_2_0) {
		cout << "GLEW_VERSION_2_0 				FAILED" << endl;
		exit(1);
	} else {
		cout << "GLEW_VERSION_2_0 				OK" << endl;
	}

	if (!GLEW_ARB_vertex_program) {
		cout << "GLEW_ARB_vertex_program 			FAILED" << endl;
		exit(1);
	} else {
		cout << "GLEW_ARB_vertex_program 			OK" << endl;
	}

	if (!GLEW_ARB_fragment_program) {
		cout << "GLEW_ARB_fragment_program 			FAILED" << endl;
		exit(1);
	} else {
		cout << "GLEW_ARB_fragment_program 			OK" << endl;
	}

	if (!GLEW_ARB_texture_non_power_of_two) {
		cout << "GLEW_ARB_texture_non_power_of_two		FAILED" << endl;
		exit(1);
	} else {
		cout << "GLEW_ARB_texture_non_power_of_two 		OK" << endl;
	}
}

void* Renderer::generateTimestepTextures(void* params)
{
	int threadIndex = *((int*) params);

	int dimX = Renderer::instance->dataset->getDimX();
	int dimY = Renderer::instance->dataset->getDimY();
	unsigned long int dimSize = (long)dimX * (long)dimY;

	FlowGeometry& geometry = Renderer::instance->dataset->getGeometry();
	float geomMinX = geometry.getMinX();
	float geomMinY = geometry.getMinY();
	float geomMaxX = geometry.getMaxX();
	float geomMaxY = geometry.getMaxY();

	int timestepsPerThread = Renderer::instance->dataset->getNumTimesteps() / MAX_PREPROCESSING_THREADS;
	if (Renderer::instance->dataset->getNumTimesteps() % MAX_PREPROCESSING_THREADS != 0)
		timestepsPerThread++;

	int startTimestamp = threadIndex * timestepsPerThread;
	int endTimestamp = ((threadIndex + 1) * timestepsPerThread) - 1;
	if (endTimestamp >= Renderer::instance->dataset->getNumTimesteps())
		endTimestamp = Renderer::instance->dataset->getNumTimesteps() - 1;
	
	cout << "Thread " << threadIndex << " will calculate Timestamps from " << startTimestamp << " to " << endTimestamp << endl;

	for (int t = startTimestamp; t <= endTimestamp; t++) {
		vector<float*> channelData;
		TimeChannel* timeStep = Renderer::instance->dataset->getTimeChannel(t);

		cout << "Processing timestep " << t << " ..." << endl;
		
		for (int ch = 0; ch < Renderer::instance->dataset->getChannelsNum(); ch++) {
			//std::cout << "t: " << t << "  ch: " << ch << std::endl;
			FlowChannel* channel = timeStep->getChannel(ch);
			float* data = (float*) malloc(dimSize * sizeof(float));

			for (int y = 0; y < dimY; y++) {
				for (int x = 0; x < dimX; x++) {
					int index = y * dimX + x;
					float xPos = ((float) x / (float) dimX) * geomMaxX; //* (geomMaxX - geomMinX) + geomMinX;
					float yPos = ((float) y / (float) dimY) * geomMaxY; //* (geomMaxY - geomMinY) + geomMinY;
					vec3 pos(xPos, yPos);
					//std::cout << index << ", x:" << x << ", y:" << y << ", max:" << dimSize << " (" << dimX << "," << dimY << ")" << std::endl;
					data[index] = channel->normalizeValue(channel->getValue(index));
				}
			}

			channelData.push_back(data);
		}


		// MAGNITUDE CALCULATION

		/*
		FlowChannel* channelX = timeStep->getChannel(0);
		FlowChannel* channelY = timeStep->getChannel(1);
		FlowChannel* channelZ = timeStep->getChannel(2);
		float* data = (float*) malloc(dimSize * sizeof(float));
	
		for (int y = 0; y < dimY; y++)
		{
			for (int x = 0; x < dimY; x++)
			{
				int index = y * dimY + x;
				float xPos = ((float) x / (float) dimX) * (geomMaxX - geomMinX) + geomMinX;
				float yPos = ((float) y / (float) dimY) * (geomMaxY - geomMinY) + geomMinY;
				vec3 pos(xPos, yPos);
			
				vec3 dataVec = vec3(channelX->normalizeValue(channelX->getValue(pos)),
						    channelY->normalizeValue(channelY->getValue(pos)),
						    channelZ->normalizeValue(channelZ->getValue(pos)));
				data[index] = channelX->normalizeValue(dataVec.length());
			}
		}

		channelData.push_back(data);
		*/

		sem_wait(&instance->mutex);
		Renderer::instance->timeStepData[t] = channelData;
		sem_post(&instance->mutex);
	}

	cout << "Generating ChannelTextures finished" << endl;

	delete (int*) params;

	pthread_exit(0);
}
