/****************************************************************************
**
** Copyright (C) 2005-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the demonstration applications of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
** 
** Trolltech ASA (c) 2007
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef HOVERPOINTS_H
#define HOVERPOINTS_H

#include <QWidget>
#include <QCheckBox>
#include <QComboBox>
#include <QLabel>
#include <QRadioButton>
#include <QButtonGroup>
#include <QPushButton>
#include <QLineEdit>
#include <QSlider>
#include <QDomDocument>
#include <QDomElement>
#include <QPen>
#include <QColorDialog>

#include <iostream>

#include "../renderer/Renderer.h"

struct TFPnt
{
	QPointF pos;
	QColor col;
};

/**
* @class Hoverpoints provides functions for manipulating points in the transferfunction editor
*/
class HoverPoints : public QObject
{
    Q_OBJECT
public:
    enum PointShape {
        CircleShape,
        RectangleShape
    };

    enum LockType {
        LockToLeft   = 0x01,
        LockToRight  = 0x02,
        LockToTop    = 0x04,
        LockToBottom = 0x08
    };

    enum SortType {
        NoSort,
        XSort,
        YSort
    };

    enum ConnectionType {
        NoConnection,
        LineConnection,
        CurveConnection
    };

    HoverPoints(QWidget *widget, PointShape shape);

    //! @brief defines what actions to be taken on which event
    bool eventFilter(QObject *object, QEvent *event);

    //! @brief draw points and lines between points
    void paintPoints();

    //! @brief used to check if the mouseclick was near the point
    inline QRectF boundingRect() const;
    //! @brief used to define the size of the area where a point is located
    void setBoundingRect(const QRectF &boundingRect) { m_bounds = boundingRect; }

    QVector<TFPnt> points() const { return m_points; }
    void setPoints(const QVector<TFPnt> &points);

    QSizeF pointSize() const { return m_pointSize; }
    void setPointSize(const QSizeF &size) { m_pointSize = size; }

    //! @brief X or Y direction
    SortType sortType() const { return m_sortType; }
    //! @brief X or Y direction
    void setSortType(SortType sortType) { m_sortType = sortType; }

    //! @brief linear or cubic
    ConnectionType connectionType() const { return m_connectionType; }
    //! @brief linear or cubic
    void setConnectionType(ConnectionType connectionType) { m_connectionType = connectionType; }

    //! @brief used for drawing
    void setConnectionPen(const QPen &pen) { m_connectionPen = pen; }
    //! @brief used for drawing
    void setShapePen(const QPen &pen) { m_pointPen = pen; }
    //! @brief used for drawing
    void setShapeBrush(const QBrush &brush) { m_pointBrush = brush; }

    //! @brief set points that can only be changed in specific directions
    void setPointLock(int pos, LockType lock) { m_locks[pos] = lock; }

    void setEditable(bool editable) { m_editable = editable; }
    bool editable() const { return m_editable; }


 signals:
    //! @brief emitted if points have changed
    void pointsChanged(const QVector<TFPnt> &points);

public slots:
    void setEnabled(bool enabled);
    void setDisabled(bool disabled) { setEnabled(!disabled); }

public:
    void firePointChange();

private:
    inline QRectF pointBoundingRect(int i) const;
    void movePoint(int i, const QPointF &newPos, bool emitChange = true);

    QWidget *m_widget;

    //QPolygonF m_points;
    QVector<TFPnt> m_points;
    QRectF m_bounds;
    PointShape m_shape;
    SortType m_sortType;
    ConnectionType m_connectionType;

    QVector<uint> m_locks;
    QVector<QColor> m_colors;

    QSizeF m_pointSize;
    int m_currentIndex;
    bool m_editable;
    bool m_enabled;

    QPen m_pointPen;
    QBrush m_pointBrush;
    QPen m_connectionPen;
};


inline QRectF HoverPoints::pointBoundingRect(int i) const
{
    QPointF p = m_points.at(i).pos;
    double w = m_pointSize.width();
    double h = m_pointSize.height();
    double x = p.x() - w / 2;
    double y = p.y() - h / 2;
    return QRectF(x, y, w, h);
}

inline QRectF HoverPoints::boundingRect() const
{
    if (m_bounds.isEmpty())
    {
        return m_widget->rect();
	
    }
    else
        return m_bounds;
}

#endif
