#include "Transfer.h"

/**
* Transfer constructor.
* This is the class of the transfer function which handles color interpolation between selected
* colors at certain densities. DensityArray contains the occurrences of densities. num defines
* the amount of density values used.
*
* @param std::vector<int> DensityArray
* @param int num
*/
Transfer::Transfer (std::vector<int> DensityArray, int num)
{
	DensityArray.swap(hist);
	newColors.resize(hist.size());
	numP = num;
	colorf = new float[numP*4];
	changed = true;
}

 /**
   * Add new points to transfer function. This function adds new points to a list of points
   * which is then used to interpolate their corresponding color values.
   * 
   * @param DensityPoint p
   */
void Transfer::addPoint(DensityPoint* p)
{
	points.push_back(p);
	sortPoints();
	changed = true;
	colorInterpolation();
}

/**
* Remove points from transfer function. This function removes the point from the list of points
* at a given position..
* 
* @param int pos
*/
void Transfer::removePoint(int pos)
{
	points.erase(points.begin()+pos,points.end()-(points.size()-pos-1));
	sortPoints();
	colorInterpolation();
	changed = true;
}

/**
* Remove ALL points from transfer function. This function is mainly used for
* cleanup.
* 
*/
void Transfer::removePoints()
{
	points.erase(points.begin(),points.end());
}

/**
* Sort the list of points by density. To ensure that the list can be used for
* interpolation, the list first has to be sorted. They are sorted in descending
* order corresponding to their density value.
* 
*/
void Transfer::sortPoints() 
{
	DensityPoint* p;
	
	unsigned int i = 0;
	while(!points.empty() && i < points.size()-1) 
	{
		if(points[i]->getDensity()>points[i+1]->getDensity()) 
		{
				p = points[i];
				points[i] = points[i+1];
				points[i+1] = p;
				i=0;
		}
		else
			i++;
	}
}

/**
* Color interpolation for the transfer function. The colors for the texture are interpolated from
* the points in the list. The points are obtained from the transfer function on the user inteface.
* The program initializes with two points in the list, which are created automatically to ensure a
* functioning interpolation. These are a white point at density = 1 and a black point at density = 0.
* This function returns the color vector (containing floats) of length num*4 for a texture.
*
*@return float* colorf
*/
float* Transfer::colorInterpolation() {

	Color* c;
	if (changed)
		changed = false;

	//if first point is not at density = 0, make a new entry in list with density 0 and color black	
	if(points.empty() || points[0]->getDensity()!= 0) 
	{
		c = new Color(0.0f,0.0f,0.0f,0.0f);
		DensityPoint* first = new DensityPoint(0.0f,c);
		std::vector<DensityPoint*> copy(points.size());
		
		copy = points;
		points.resize(points.size()+1);
		points[0] = first;

		for (int i = 0; i < (int)copy.size(); i++)
		{
			points[i+1] = copy[i];
		}
	}
	else //should never happen, but just in case
		c = points[0]->getColor();

	newColors[0] = c;

	//if last point is not at density = 1, make a new entry in list with density 1 and color white
	if(points[points.size()-1]->getDensity()!=1)
	{
		c = new Color(1.f,1.f,1.f,1.f);
		DensityPoint* last = new DensityPoint(1.f,c);
		points.resize(points.size()+1);
		points[points.size()-1] = last;
	}
	else
		c = points[points.size()-1]->getColor();

	newColors[newColors.size()-1] = c;

	DensityPoint* prevPoint;
	Color* prevColor;
	DensityPoint* currentPoint;
	Color* currentColor;
	float rd; float gd; float bd; float ad;
	float rPart; float gPart; float bPart; float aPart;
	float R; float G; float B; float A;
	int nPrev; int nCurr;

	//Interpolation for all colors between the defined points 
	for (int i = 0; i < (int)points.size()-1; i++)
	{	
		rd = 0; gd = 0; bd = 0; ad = 0;
		rPart = 0; gPart = 0; bPart = 0; aPart = 0;
		R = 0; G = 0; B = 0; A = 0;

		//First point with user-defined color
		prevPoint = points[i];
		prevColor = prevPoint->getColor();

		//Second point with user-defined color
		currentPoint = points[i+1];
		currentColor = currentPoint->getColor();

		//Distance of current point to previous point
		float d = currentPoint->getDensity()*numP - prevPoint->getDensity()*numP;

		if (d != 0)
		{
			nPrev = (int)(prevPoint->getDensity()*numP);
			nCurr = (int)(currentPoint->getDensity()*numP)-1;

			//Assign colors of user-defined points to densities
			newColors[nPrev] = prevColor;
			newColors[nCurr] = currentColor;

			//Get the amount of each channel to be added in each step to densities in between
			rd = currentColor->GetNormalizedRed() - prevColor->GetNormalizedRed();
			gd = currentColor->GetNormalizedGreen() - prevColor->GetNormalizedGreen();
			bd = currentColor->GetNormalizedBlue() - prevColor->GetNormalizedBlue();
			ad = currentColor->GetNormalizedAlpha() - prevColor->GetNormalizedAlpha();

				rPart = rd/d;
				gPart = gd/d;
				bPart = bd/d;
				aPart = ad/d;

			R = prevColor->GetNormalizedRed();
			G = prevColor->GetNormalizedGreen();
			B = prevColor->GetNormalizedBlue();
			A = prevColor->GetNormalizedAlpha();
			//Step through the densities in between the previous and current point
			for (int j = nPrev+1; j < nCurr; j++)
			{
				//Calculate the new color for each density
				R = R + rPart;
				G = G + gPart;
				B = B + bPart;
				A = A + aPart;
				c = new Color(R,G,B,A);
				newColors[j] = c;
			}
		}
	}

	colorf = getColorf();
	return colorf;
}

std::vector<Color*> Transfer::getColors(){
	return newColors;
}

/**
* Get the color vector. This function returns a vector of interpolated
* colors of size num*4.
* 
* @return float* colorf
*/
float* Transfer::getColorf()
{
	//colorInterpolation();

	int j = 0;
	for (int i = 0;  i < newColors.size(); i++)
	{
		colorf[j] = newColors[i]->GetNormalizedRed(); j++;
		colorf[j] = newColors[i]->GetNormalizedGreen(); j++;
		colorf[j] = newColors[i]->GetNormalizedBlue(); j++;
		colorf[j] = newColors[i]->GetNormalizedAlpha(); j++;

		//if(i == 300)
		//	int a = 5;
	}

	return colorf;
}

/**
* Clear data in transfer function. This function is used to clean up before
* loading new data. The empty color vector colorf is returned.
* 
* @return float* colorf
*/
float* Transfer::clearData()
{
	//newColors
	newColors.clear();
	points.clear();
	for(int i = 0; i < numP*4; i++)
		colorf[i]=0.0f;

	return colorf;
}
