//------------------------------------
float4x4 worldViewProjMat;				// World View Projection Matrix
float4x4 worldViewIT;					// InverseTransformed ViewProjection Matrix
//float4 zVec;

texture3D volumeTexture;				//Volume Texture
texture frontFaceTexture;				//FrontFaceTexture
texture backFaceTexture;				//BackFaceTexture
texture1D transferFunctionTexture;		//TransferfunctionTexture

bool backFace;							//Back or Frontface?
bool filterNearest;						//filterNearest?

float stepSize = 1.0f;					//stepSize for raycasting
float alphaFactor = 1.0f;				//Correctionvalue for the alphachannel
int maxApproximation;					//Max. Approximation steps
float threshold;						//Firsthit threshold
bool useTransferFunction;				//Should use transferFunction
float slicePercent;						//SliceId in range of 0....1
float viewMin;							//min Point
float viewMax;							//max Point
int iProjection;						//ProjectionMode

float nearPlane;						//NearPlane

float4 lightDir;						//LightDir
float4 halfVector;						//HalfVector

float4 ambient;							//Ambient color
float4 diffuse;							//Diffuse color
float4 specular;						//Specular color
float  power;							//Specular power

//------------------------------------
/**
Vertex Input
*/
struct vertexInput {
    float3 position				: POSITION;
    float4 texCoord				: TEXCOORD0;
};

/**
Vertex Output for slices
*/
struct vertexOutputSlice {
	float4 position				: POSITION;
	float4 texCoord				: TEXCOORD0;
	float4 pos					: TEXCOORD1;
};

/**
Vertex Output
*/
struct vertexOutput {
	float4 position				: POSITION;
	float4 texCoord				: TEXCOORD0;
};

/**
Pixel Ouput
*/
struct pixelOutput
{
	float4 color				: COLOR;
};

//------------------------------------
/**
Linear sampler template
*/
#define SAMPLER_LINEAR(g_samplerMap, g_txMap);	\
	sampler2D g_samplerMap = sampler_state {	\
    Texture = <g_txMap>;						\
    MinFilter = Linear;							\
    MagFilter = Linear;							\
    MipFilter = Linear;							\
    AddressU  = BORDER;							\
    AddressV  = BORDER;							\
};

/**
Point sampler template
*/
#define SAMPLER_POINT(g_samplerMap, g_txMap);	\
	sampler2D g_samplerMap = sampler_state {	\
    Texture = <g_txMap>;						\
    MinFilter = Point;							\
    MagFilter = Point;							\
    MipFilter = Point;							\
    AddressU  = BORDER;							\
    AddressV  = BORDER;							\
};

/**
Linear volume sampler
*/
sampler3D samplerVolTexLin = sampler_state
{
	Texture = <volumeTexture>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
    AddressU = CLAMP;
    AddressV = CLAMP;
};

/**
Point volume sampler
*/
sampler3D samplerVolTexPnt = sampler_state
{
	Texture = <volumeTexture>;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = Point;
    AddressU = CLAMP;
    AddressV = CLAMP;
};

/**
Sampler for the transferFunction
*/
sampler1D samplerTransferFunction = sampler_state
{
	Texture = <transferFunctionTexture>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
    AddressU = CLAMP;
    AddressV = CLAMP;
};

/**
Linear Sampler for frontFaceTexture
*/
SAMPLER_LINEAR(samplerFrontFace, frontFaceTexture);
/**
Linear Sampler for backFaceTexture
*/
SAMPLER_LINEAR(samplerBackFace, backFaceTexture);

//----------------------------------------------

/**
EntryPoint VertexShader
*/
vertexOutputSlice EntryPointVS(vertexInput IN) 
{
    vertexOutputSlice OUT;
    float4 vertexpos = float4(IN.position.xyz, 1.0);
	float4 eyespace = mul(vertexpos, worldViewProjMat);
	OUT.position = eyespace;
	OUT.pos = eyespace;
	OUT.texCoord = float4(IN.texCoord.xyz, eyespace.z);
	return OUT;
}

/**
EntryPoint PixelShader
*/
pixelOutput EntryPointPS(vertexOutputSlice IN)
{
	pixelOutput OUT;
	if(backFace) {
		OUT.color = IN.texCoord;
	} else {
		if(IN.texCoord.w<viewMin+(viewMax-viewMin)*slicePercent) {
			//samplerBackFace
			float4 exitPoint = tex2D(samplerBackFace, (float2(IN.pos.x,-IN.pos.y)/IN.pos.w*0.5+0.5));
			float4 entryPoint = IN.texCoord;
			
			float3 dist = exitPoint.xyz - entryPoint.xyz;
			float l = length(dist);
			
			float frontDist = entryPoint.w - viewMin;
			float backDist = viewMax - exitPoint.w;
			float frontFactor = (entryPoint.w - viewMin)/(viewMax - viewMin);
			float backFactor = (exitPoint.w-viewMin)/(viewMax - viewMin);
			
			float f = min(slicePercent, backFactor);
			f = max(f, frontFactor);
			
			f -= frontFactor;
			f *= 1/(backFactor-frontFactor);
			OUT.color = f*exitPoint + (1-f)*entryPoint;
		} else {
			OUT.color = IN.texCoord;
		}
	}
	return OUT;
}

/**
Vertexshader for slices
*/
vertexOutput SliceVS(vertexInput IN) 
{
    vertexOutput OUT;
    float4 vertexpos = float4(IN.position.xyz, 1.0);
	float4 eyespace = mul(vertexpos, worldViewProjMat);
	OUT.position = eyespace;
	OUT.texCoord = float4(IN.texCoord.xyz, 0);
	return OUT;
}

/**
Pixelshader for slices
*/
pixelOutput SlicePS(vertexOutput IN) 
{
	pixelOutput OUT;
	float value;
	if(filterNearest) {
		value = tex3Dlod(samplerVolTexPnt, float4(IN.texCoord.xyz,1)).w;
	} else {
		value = tex3Dlod(samplerVolTexLin, float4(IN.texCoord.xyz,1)).w;
	}
	
	if(useTransferFunction) {
		OUT.color = tex1D(samplerTransferFunction, value);
	} else {
		OUT.color = float4(value, value, value,1);
	}
	return OUT;
}

/**
FirstHitFunction
*/
void firstHit(in float4 rayDir,
			  in float4 exitPoint,
			  inout float4 value,
			  inout int approxCounter,
			  inout float4 rayPos) {
			  
	//Iterate through Volume
	for(int i=0;i<1024 && approxCounter<maxApproximation && rayPos.w<exitPoint.w;i++) {
		if(filterNearest) {
			value = tex3Dlod(samplerVolTexPnt, float4(rayPos.xyz,1));
		} else {
			value = tex3Dlod(samplerVolTexLin, float4(rayPos.xyz,1));
		}
		
		//While Value below threshold go straight forward
		if(value.w<threshold && approxCounter==1) {
			rayPos+=rayDir;
		} else {
			//Value aboth/below threshold. Take half stepsize and approximate near real point
			approxCounter *= 2;
			rayPos+=(1.0f/approxCounter)*rayDir*sign(threshold-value.w);
		}
	}
}

/**
Averagefunction
*/
void average(in float4 rayDir,
			 in float4 exitPoint,
			 inout float4 dstColor,
			 inout float4 rayPos) {
		
	float value;
	for(int i=0;i<1023 && rayPos.w<exitPoint.w;i++) {
		//Choose filtering
		if(filterNearest) {
			value = tex3Dlod(samplerVolTexPnt, float4(rayPos.xyz, 1)).w;
		} else {
			value = tex3Dlod(samplerVolTexLin, float4(rayPos.xyz, 1)).w;
		}
		
		//Should use Transferfunction?
		if(useTransferFunction) {
			dstColor += tex1Dlod(samplerTransferFunction, value);
		} else {
			dstColor += float4(value, value, value,1);
		}
		rayPos += rayDir;
	}
}

/**
Maximum function
*/
void maximum(in float4 rayDir,
			 in float4 exitPoint,
			 inout float maxValue,
			 inout float4 rayPos) {
	
	float value;	 
	for(int i=0;i<1024 && rayPos.w<exitPoint.w;i++) {
		if(filterNearest) {
			value = tex3Dlod(samplerVolTexPnt, float4(rayPos.xyz, 1)).w;
		} else {
			value = tex3Dlod(samplerVolTexLin, float4(rayPos.xyz, 1)).w;
		}
		if(value>maxValue) {
			maxValue = value;
		}
		rayPos += rayDir;
	}
}

/**
Levoy function
*/
void levoy(in float4 rayDir,
		   in float4 exitPoint,
		   inout float4 dstColor,
		   inout float4 rayPos) {
	
	float4 src;
	float4 value;
	float3 normal;
	for(int i=0;i<1024 && rayPos.w<exitPoint.w;i++) {
		//Choose filtering
		if(filterNearest) {
			value = tex3Dlod(samplerVolTexPnt, float4(rayPos.xyz, 1));
		} else {
			value = tex3Dlod(samplerVolTexLin, float4(rayPos.xyz, 1));
		}
		src = tex1Dlod(samplerTransferFunction, value.w);
		
		//Calculate Lightning
		normal = value.xyz*2.0f-1.0f;
		
		//Specular
		float3 specI = specular*pow(clamp(0, 1, dot(normal, halfVector)), power);
		
		//Diffuse
		float3 diffI = diffuse*clamp(0,1,dot(normal, lightDir));
		
		src.rgb = src.rgb*(diffI + specI + ambient.xyz);
		src.a *= alphaFactor;
		
		dstColor = dstColor + (1-dstColor.a)*float4(src.rgb*src.a, src.a);
		
		if(dstColor.a>0.95) {
			rayPos.w = exitPoint.w+1.0f;
		}
		
		rayPos += rayDir;
	}
}

/**
Firsthit Vertexshader
*/
vertexOutput RayCasterFHVS(vertexInput IN) 
{
    vertexOutput OUT;
	OUT.position = float4(IN.position.xyz, 1.0);
	OUT.texCoord = IN.texCoord;
	return OUT;
}

/**
Firsthit Pixelshader
*/
pixelOutput RayCasterFHPS(vertexOutput IN) 
{
	pixelOutput OUT;
	
	float4 dstColor = float4(0, 0, 0, 0);
	float3 normal;
	float4 value = 0;
	
	//Read out entry and exitpoints
	float4 entryPoint = tex2D(samplerFrontFace, IN.texCoord.xy);
	float4 exitPoint = tex2D(samplerBackFace, IN.texCoord.xy);
	
	//Calculate raydirection and starting raypos
	float4 rayDir = normalize(exitPoint - entryPoint);
	float4 rayPos = entryPoint;
	
	if((iProjection==22 && length(rayDir)<0.4f) || (iProjection==23 && length(rayDir)==0.0f)) {
		//No volume to render
		OUT.color = float4(0, 0, 0, 0);
	} else  {
		//Calculate Stepsize
		rayDir *= stepSize;
		int approxCounter = 1;
		int callCounter = 0;
		
		//Iterate through Volume
		firstHit(rayDir, exitPoint, value, approxCounter, rayPos);
		while(callCounter<10 && approxCounter<maxApproximation && rayPos.w<exitPoint.w) {
			firstHit(rayDir, exitPoint, value, approxCounter, rayPos);
			callCounter++;
		}
		
		//if a threshold was found - calculate color
		if(approxCounter>=maxApproximation) {
			if(useTransferFunction) {
				dstColor = tex1D(samplerTransferFunction, value.w);
			} else {
				dstColor = float4(value.www, 1);
			}
		}
		
		//Do lightning Stuff
		//Since only positive values can bes stored in volume map we have to shift and scale the values for the normal vector
		normal = value.xyz*2.0f-1.0f;
		
		//Specular
		float3 specI = specular*pow(clamp(0, 1, dot(normal, halfVector)), power);
		
		//Diffuse
		float3 diffI = diffuse*clamp(0,1,dot(normal, lightDir));
		
		//Ambient = ambient
		dstColor.rgb = dstColor.rgb*(diffI + specI + ambient.rgb);
		OUT.color = dstColor;
	}
	return OUT;
}

/**
Average Vertexshader
*/
vertexOutput RayCasterAVVS(vertexInput IN) 
{
    vertexOutput OUT;
	OUT.position = float4(IN.position.xyz, 1.0);
	OUT.texCoord = IN.texCoord;
	return OUT;
}

/**
Average Pixelshader
*/
pixelOutput RayCasterAVPS(vertexOutput IN) 
{
	pixelOutput OUT;
	
	float4 dstColor = float4(0, 0, 0, 0);
	float value = 0.0f;
	
	//Read out entry and exitpoints
	float4 entryPoint = tex2D(samplerFrontFace, IN.texCoord.xy);
	float4 exitPoint = tex2D(samplerBackFace, IN.texCoord.xy);
	
	float4 rayDir = exitPoint - entryPoint;
	float4 rayPos = entryPoint;
	
	if((iProjection==22 && length(rayDir)<0.4f) || (iProjection==23 && length(rayDir)==0.0f)) {
		//No volume to render
		OUT.color = float4(0, 0, 0, 0);
	} else  {
		rayDir *= stepSize;
		
		//Iterate through Volume
		int callCounter = 0;
		average(rayDir, exitPoint, dstColor, rayPos);
		while(callCounter<10 && rayPos.w<exitPoint.w) {
			average(rayDir, exitPoint, dstColor, rayPos);
			callCounter++;
		}
		
		dstColor *= stepSize;
		OUT.color = dstColor;
	}
	return OUT;
}

/**
Maximum Vertexshader
*/
vertexOutput RayCasterMAXVS(vertexInput IN) 
{
    vertexOutput OUT;
	OUT.position = float4(IN.position.xyz, 1.0);
	OUT.texCoord = IN.texCoord;
	return OUT;
}

/**
Maximum Pixelshader
*/
pixelOutput RayCasterMAXPS(vertexOutput IN) 
{
	pixelOutput OUT;
	float value = 0;
	float maxValue = 0;
	
	//Read out entry and exitpoints
	float4 entryPoint = tex2D(samplerFrontFace, IN.texCoord.xy);
	float4 exitPoint = tex2D(samplerBackFace, IN.texCoord.xy);
	
	float4 rayDir = normalize(exitPoint - entryPoint);
	float4 rayPos = entryPoint;
	
	if((iProjection==22 && length(rayDir)<0.4f) || (iProjection==23 && length(rayDir)==0.0f)) {
		//No volume to render
		OUT.color = float4(0, 0, 0, 0);
	} else  {		
		rayDir *= stepSize;
		
		int callCounter = 0;
		maximum(rayDir, exitPoint, maxValue, rayPos);
		while(callCounter<10 && rayPos.w<exitPoint.w) {
			maximum(rayDir, exitPoint, maxValue, rayPos);
			callCounter++;
		}
		
		
		if(useTransferFunction) {
			OUT.color = tex1D(samplerTransferFunction, maxValue);
		} else {
			OUT.color = float4(maxValue, maxValue, maxValue, 1);
		}
	}
	return OUT;
}

/**
Levoy Vertexshader
*/
vertexOutput RayCasterLEVOYVS(vertexInput IN) 
{
    vertexOutput OUT;
	OUT.position = float4(IN.position.xyz, 1.0);
	OUT.texCoord = IN.texCoord;
	return OUT;
}

/**
Levoy Pixelshader
*/
pixelOutput RayCasterLEVOYPS(vertexOutput IN) 
{
	pixelOutput OUT;
	
	float4 dstColor = float4(0, 0, 0, 0);
	
	//Read out entry and exitpoints
	float4 entryPoint = tex2D(samplerFrontFace, IN.texCoord.xy);
	float4 exitPoint = tex2D(samplerBackFace, IN.texCoord.xy);
	
	float4 rayDir = normalize(exitPoint - entryPoint);
	float4 rayPos = entryPoint;
	
	if((iProjection==22 && length(rayDir)<0.4f) || (iProjection==23 && length(rayDir)==0.0f)) {
		//No volume to render
		OUT.color = float4(0, 0, 0, 0);
	} else  {		
		rayDir *= stepSize;
		
		//Iterate through Volume
		int callCounter = 0;
		levoy(rayDir, exitPoint, dstColor, rayPos);
		while(callCounter<10 && rayPos.w<exitPoint.w) {
			levoy(rayDir, exitPoint, dstColor, rayPos);
			callCounter++;
		}
		
		OUT.color = dstColor;
	}
	return OUT;
}


// 
// Effect
//

#define Technique(name);								\
	technique name										\
	{													\
	    pass p0											\
	    {												\
		    VertexShader = compile vs_3_0 name##VS();	\
		    PixelShader  = compile ps_3_0 name##PS();	\
		}												\
	}

Technique( Slice );
Technique( EntryPoint );
Technique( RayCasterFH );
Technique( RayCasterMAX );
Technique( RayCasterAV );
Technique( RayCasterLEVOY );
